#include "copyright.i"

!*******************************************************************************!
! Module: amoeba_runmd_mod
!
! Description: <TBS>
!
!*******************************************************************************

module amoeba_runmd_mod
#ifdef AMOEBA

implicit none

private

  double precision, parameter :: kcal_to_gm_x_Ang2_per_ps2 = 4.184d+2
  double precision, parameter :: ideal_gas_const_kcal_per_mol_K = 1.9872065d-3
  double precision, parameter :: tinker_to_sander_time_convert = 1.d0 / 20.455d0

  public        beeman_runmd

contains

!*******************************************************************************!
! Subroutine:  beeman_runmd
!
! Description: <TBS>
!
!*******************************************************************************

#ifdef MPI
subroutine beeman_runmd(atm_cnt, crd, mass, frc, vel, my_atm_lst)
#else
subroutine beeman_runmd(atm_cnt, crd, mass, frc, vel)
#endif

  use cit_mod
  use dynamics_dat_mod
  use file_io_dat_mod
  use file_io_mod
  use img_mod
  use inpcrd_dat_mod
  use mdin_ctrl_dat_mod
  use mdin_ewald_dat_mod
  use nb_pairlist_mod
  use nmr_calls_mod
  use parallel_dat_mod
  use parallel_mod
  use pbc_mod
  use prmtop_dat_mod
  use runfiles_mod
  use state_info_mod
  use amoeba_force_mod
  use loadbal_mod
  use timers_mod
  use constraints_mod

  implicit none

! Formal arguments:

  integer, intent(in)                   :: atm_cnt
  double precision, intent(in out)      :: crd(3, *)
  double precision, intent(in)          :: mass(*)
  double precision, intent(in out)      :: frc(3, *)
  double precision, intent(in out)      :: vel(3, *)
#ifdef MPI
  integer                               :: my_atm_lst(*)
#endif

! Local variables:

  logical                               :: new_list
  logical                               :: write_mdcrd
  logical                               :: write_restrt
  type(amba_pot_ene_rec)                :: amba_pot_ene

  integer                               :: m, nstep
#ifdef MPI
  integer                               :: atm_lst_idx
  double precision                      :: new_list_cnt
  double precision, save                :: reduce_buf_in(3)
  double precision, save                :: reduce_buf_out(3)
#endif
  double precision                      :: fac(3)
  double precision                      :: fac_inv(3), degrees_of_freedom
  double precision                      :: si(si_cnt)
  double precision                      :: sit(si_cnt), sit_tmp(si_cnt)
  double precision                      :: sit2(si_cnt), sit2_tmp(si_cnt)

! from am_runmd_beeman_md_step():

  double precision                      :: kinetic_energy(3, 3)
  double precision                      :: eksum
  double precision                      :: convert
  double precision                      :: pressure
  double precision                      :: temperature
  double precision                      :: virial(3)

! from am_runmd_remove_com():

  integer                               :: n
  double precision                      :: vcm(3)
  double precision                      :: amass, tmassinv, vel2, atempdrop

! from am_runmd_running_ave():

! from am_runmd_beeman_1st_crd_update():

  integer                               :: j
  double precision                      :: term(3), delt_8, delt2_8

! from am_runmd_kinetic_energy_calc():

  integer                               :: k
  double precision                      :: calc_term

! from am_runmd_berendson_scaling():

  double precision                      :: scale_factor

! from am_runmd_pressure():

  double precision, parameter           :: pressure_constant = 6.85695d+4

  call zero_time()

  degrees_of_freedom = dble(3 * atm_cnt - 3)! num crds minus overall translation
                                            ! FIXME change if constraints

  fac_inv(1) = 2.d0 / (degrees_of_freedom * ideal_gas_const_kcal_per_mol_K)
  fac_inv(2) = 1.d-6
  fac_inv(3) = 1.d-6

  fac(1) = 1.d0 / fac_inv(1)
  fac(2) = 1.d0 / fac_inv(2)
  fac(3) = 1.d0 / fac_inv(3)

  ! clear the energy array

  si(:) = 0.d0
  sit(:) = 0.d0
  sit2(:) = 0.d0

  ! The _tmp variables below are only used if ntave .gt. 0.  They are used for
  ! scratch space in the ntave calcs, and to hold the last sit* values between
  ! calls to the ntave code.

  sit_tmp(:) = 0.d0
  sit2_tmp(:) = 0.d0

  if (ntp .gt. 0) then
    si(si_volume) = uc_volume
    si(si_density) = tmass / (0.602204d0 * si(si_volume))
  end if

  if (ntpr .le. 0) ntpr = nstlim

  if (master) call amopen(mdinfo, mdinfo_name, 'U', 'F', 'W')

  delt_8 = dt / 8.0d0
  delt2_8 = dt * delt_8

  do nstep = 1, nstlim

    ! Begin beeman_md_step:

    ! Begin beeman_1st_crd_update:

! In the current implementation, all the crds, vels, accels are updated on
! all nodes in every step for mpi.  This is horrendously inefficient, but does
! not matter until the ton of other inefficiencies are fixed, and it is
! absolutely safe. We update the "old" versions by a copy step, also not
! efficient but absolutely safe...
! In order to maintain consistency with sander amoeba, we do some gather's to
! the master that are also inefficient.  Unfortunately, this integrator was
! laid out in a manner that makes parallel efficiency all but impossible.  This
! is not at all necessary, and should be fixed, but when it is fixed it will
! produce slightly different results (basically dependent on where in the step
! various updates and sampling occurs).  If we care about Beeman integrator
! efficiency, we should produce an alternative implementation; these are
! purely matters of thinking about parallelization issues in advance, not
! issues of correctness.

#ifdef MPI
    do atm_lst_idx = 1, my_atm_cnt
      n = my_atm_lst(atm_lst_idx)
#else
    do n = 1, atm_cnt
#endif
      do j = 1, 3
        term(j) = 5.d0 * acceleration(j, n) - old_acceleration(j, n)
        crd(j, n) = crd(j, n) + vel(j, n) * dt + term(j) * delt2_8
        vel(j, n) = vel(j, n) + term(j) * delt_8
      end do
    end do

    ! End beeman_1st_crd_update.

    if (nstep .gt. 1) then

#ifdef MPI
      call check_my_atom_movement(crd, gbl_atm_saved_crd, my_atm_lst, &
                                  skinnb, ntp, new_list)
      if (new_list) then
        new_list_cnt = 1.d0
      else
        new_list_cnt = 0.d0
      end if

      call update_time(runmd_time)
      reduce_buf_in(1) = new_list_cnt
      call mpi_allreduce(reduce_buf_in, reduce_buf_out, 1, &
                         mpi_double_precision, mpi_sum, mpi_comm_world, &
                         err_code_mpi) 
      new_list_cnt = reduce_buf_out(1)
      call update_time(fcve_dist_time)

      ! Determine if any process saw an atom exceed the skin check.  We use the
      ! comparison to 0.5d0 to handle any rounding error issues; we use
      ! double precision for new_list_cnt in order to be able to piggyback the
      ! reduce.

      new_list = (new_list_cnt .ge. 0.5d0)

      call check_new_list_limit(new_list)       ! useful later in loadbalancing

#else
      call check_all_atom_movement(atm_cnt, crd, gbl_atm_saved_crd, skinnb, &
                                   ntp, new_list)
#endif

    else
      new_list = .true. ! Do list update on first call to amoeba_force...
    end if

    ! call rattle here if used---future
    ! get the forces

    call update_time(runmd_time)

#ifdef MPI
    ! Inefficient all coordinates update...
    call mpi_allgathervec(atm_cnt, crd)
    call update_time(fcve_dist_time)
#endif

#ifdef MPI
! BUGBUG - Loadbalancing not yet supported for amoeba...
!       call do_load_balancing(new_list, atm_cnt)
    call amoeba_force(atm_cnt, crd, frc, gbl_img_atm_map, gbl_atm_img_map, &
                      my_atm_lst, new_list, amba_pot_ene, &
                      si(si_diprms), si(si_dipiter), virial)
#else
    call amoeba_force(atm_cnt, crd, frc, gbl_img_atm_map, gbl_atm_img_map, &
                      new_list, amba_pot_ene, &
                      si(si_diprms), si(si_dipiter), virial)
#endif

    ! Store energy terms in state info array for printout.

    si(si_pot_ene) = amba_pot_ene%total
    si(si_vdw_ene) = amba_pot_ene%vdw
    si(si_elect_ene) = amba_pot_ene%elec
    si(si_hbond_ene) = amba_pot_ene%hbond
    si(si_bond_ene) = amba_pot_ene%bond
    si(si_angle_ene) = amba_pot_ene%angle
    si(si_dihedral_ene) = amba_pot_ene%dihedral
    si(si_vdw_14_ene) = amba_pot_ene%vdw_14
    si(si_elect_14_ene) = amba_pot_ene%elec_14
    si(si_restraint_ene) = amba_pot_ene%restraint
    si(si_polar) = amba_pot_ene%polar

    si(si_tot_virial) = virial(1) + virial(2) + virial(3)

    ! get the next accelerations from forces

    ! Begin beeman_full_step_vels:

    ! Save all the old accelerations; not really necessary unless atoms
    ! are reassigned.

    old_acceleration(:,:) = acceleration(:,:)

#ifdef MPI
    do atm_lst_idx = 1, my_atm_cnt
      n = my_atm_lst(atm_lst_idx)
#else
    do n = 1, atm_cnt
#endif
      do j = 1, 3

        acceleration(j, n) = kcal_to_gm_x_Ang2_per_ps2 * &
                             frc(j, n) * atm_mass_inv(n)

        vel(j, n) = vel(j, n) + &
                    (3.d0 * acceleration(j,n) + old_acceleration(j,n)) * delt_8

      end do
    end do

    ! End beeman_full_step_vels.

    ! Call rattle here again if used---future

    ! Begin kinetic_energy_calc:

    kinetic_energy(:, :) = 0.d0

#ifdef MPI
    do atm_lst_idx = 1, my_atm_cnt
      n = my_atm_lst(atm_lst_idx)
#else
    do n = 1, atm_cnt
#endif
      do k = 1, 3
        calc_term = mass(n) * vel(k, n)
        kinetic_energy(:, k) = kinetic_energy(:, k) + calc_term * vel(:, n)
      end do
    end do

    ! convert units & multiply by 1/2

    calc_term = 0.5d0 / kcal_to_gm_x_Ang2_per_ps2

    do k = 1, 3
      do j = 1, 3
        kinetic_energy(j, k) = calc_term * kinetic_energy(j, k)
      end do
    end do

    eksum = kinetic_energy(1, 1) + kinetic_energy(2, 2) + kinetic_energy(3, 3)

#ifdef MPI
    call update_time(runmd_time)
    ! Sum up the partial kinetic energies. The all_reduce is expensive, and
    ! we may want to piggyback it with something else when all the other
    ! performance problems are solved (yeah, right).

    reduce_buf_in(1) = eksum
    call mpi_allreduce(reduce_buf_in, reduce_buf_out, 1, mpi_double_precision, &
                       mpi_sum, mpi_comm_world, err_code_mpi)

    eksum = reduce_buf_out(1)
    call update_time(fcve_dist_time)
#endif

    ! End kinetic_energy_calc:

    si(si_kin_ene) = eksum
    si(si_tot_ene) = si(si_kin_ene) + si(si_pot_ene)
    degrees_of_freedom = dble(3 * atm_cnt - 3) ! num crds minus overall
                                               ! translation
                                               ! FIXME change if constraints

    convert = 2.d0 / (degrees_of_freedom * ideal_gas_const_kcal_per_mol_K)
    temperature = eksum * convert

    ! Begin berendson_scaling:

    if (ntt .gt. 0) then

      ! temperature is current temperature.
      ! temp0 is target temperature.

      scale_factor = sqrt(1.d0 + (dt / tautp) * (temp0 / temperature - 1.d0))

#ifdef MPI
      do atm_lst_idx = 1, my_atm_cnt
        n = my_atm_lst(atm_lst_idx)
#else
      do n = 1, atm_cnt
#endif
        vel(:, n) = scale_factor * vel(:, n)
      end do

    end if

    ! End berendson_scaling.

    ! Here we determine if we will need all the crds in the master; if we
    ! do, we must update them after a pressure scaling step.

    write_mdcrd = .false.

    if (ntwx .gt. 0) then
      if (mod(nstep, ntwx) .eq. 0) write_mdcrd = .true.
    end if

    write_restrt = .false.
  
    if (nstep .eq. nstlim) then
      write_restrt = .true.
    else if (ntwr .ne. 0) then
      if (mod(nstep, ntwr) .eq. 0) write_restrt = .true.
    end if
    
    ! Begin pressure calc:

    if (ntp .gt. 0) then ! atom based scaling

      pressure = (pressure_constant / uc_volume) * &
                 (2.d0 * eksum - si(si_tot_virial)) / 3.d0

      ! End pressure calc.

      si(si_tot_ekcmt) = eksum  ! for printing in prntmd().
      si(si_tot_press) = pressure
      si(si_volume) = uc_volume
      si(si_density) = tmass / (0.602204d0 * si(si_volume))

      call amoeba_pressure_scale_pbc_data(pressure, dt, pres0, taup)
      call set_cit_tbl_dims(pbc_box, vdw_cutoff + skinnb, cut_factor)

#ifdef MPI
      call amoeba_pressure_scale_crds(atm_cnt, pressure, dt, pres0, taup, crd, &
                                      my_atm_lst)

      ! We may have to fix up the crds given that pressure scaling occurred...

      if (write_mdcrd .or. write_restrt) then
        call update_time(runmd_time)
        call mpi_gathervec(atm_cnt, crd)
        call update_time(fcve_dist_time)
      end if
#else
      call amoeba_pressure_scale_crds(atm_cnt, pressure, dt, pres0, taup, crd)
#endif
      if (ntr .gt. 0 .and. natc .gt. 0) then
#ifdef MPI
        call amoeba_pressure_scale_crds(atm_cnt, pressure, dt, pres0, taup, &
                                        atm_xc, my_atm_lst)

        ! We may have to fix up the crds given that pressure scaling occurred...

        if (write_mdcrd .or. write_restrt) then
          call update_time(runmd_time)
          call mpi_gathervec(atm_cnt, crd)
          call update_time(fcve_dist_time)
        end if
#else
        call amoeba_pressure_scale_crds(atm_cnt, pressure, dt, pres0, taup, &
                                        atm_xc)
#endif
      end if
    end if

    ! End beeman_md_step:

    t = t + dt

    ! Begin remove_com:

    if (nscm .gt. 0) then
      if (mod(nstep, nscm) .eq. 0) then

        vcm(:) = 0.d0

#ifdef MPI
        do atm_lst_idx = 1, my_atm_cnt
          n = my_atm_lst(atm_lst_idx)
#else
        do n = 1, atm_cnt
#endif
          amass = mass(n)
          vcm(:) = vcm(:) + amass * vel(:, n)
        end do

#ifdef MPI
        call update_time(runmd_time)
        reduce_buf_in(1:3) = vcm(:)

        call mpi_allreduce(reduce_buf_in, reduce_buf_out, 3, &
                           mpi_double_precision, &
                           mpi_sum, mpi_comm_world, err_code_mpi)

        vcm(:) = reduce_buf_out(1:3)
        call update_time(fcve_dist_time)
#endif
        tmassinv = 1.d0 / tmass
        vcm(:) = vcm(:) * tmassinv

        if (master) then
          vel2 = vcm(1) * vcm(1) + vcm(2) * vcm(2) + vcm(3) * vcm(3)
          atempdrop = 0.5d0 * tmass * vel2 * fac_inv(1)
          vel2 = sqrt(vel2)

          write (mdout, '(a, f15.6, f9.2, a)') 'check COM velocity, temp: ', &
            vel2, atempdrop, '(Removed)'
        end if

#ifdef MPI
        do atm_lst_idx = 1, my_atm_cnt
          n = my_atm_lst(atm_lst_idx)
#else
        do n = 1, atm_cnt
#endif
          vel(:, n) = vel(:, n) - vcm(:)
        end do

      end if
    end if

    ! End remove_com.

#ifdef MPI
    ! Inefficient all velocities update.  We are known to be done with changing
    ! velocities until the top of the cycle...  Also update the accelerations;
    ! pretty much unnecessary too unless there is a reassignment of 
    ! ownership...

    call update_time(runmd_time)
    call mpi_allgathervec(atm_cnt, vel)
    call mpi_allgathervec(atm_cnt, acceleration)

    ! The following update of coordinate constraints is also only necessary if
    ! there is a reassignment of atom ownership, but we are being ultra-safe
    ! here until all the issues are worked out...

    if (ntp .gt. 0) then
      if (ntr .gt. 0 .and. natc .gt. 0) then
!       if (atm_redist_needed .or. fft_slab_redist_needed) then
          call mpi_allgathervec(atm_cnt, atm_xc)
!       end if
      end if
    end if

    call update_time(fcve_dist_time)
#endif

    if (master) then

      ! Begin crd_archive:

      if (write_mdcrd) then
        call corpac(3 * atm_cnt, crd, 1, mdcrd)
        if (ntb .gt. 0) then
          call corpac(3, pbc_box, 1, mdcrd)
        end if
      end if

      ! End crd_archive.

      ! Begin vel_archive.  We know all vels are valid everywhere...

      if (ntwv .gt. 0) then
        if (mod(nstep, ntwv) .eq. 0) then
          call beeman_vel_archive(atm_cnt, vel)
        end if
      end if

      ! End vel_archive.

      ! Begin restrt write

      ! NOTE NOTE NOTE - If a run aborts before completion and
      ! the last restrt write was on an nscm step, pmemd will write a restrt
      ! with velocities consistent with those archived; ie. they include the
      ! com removal correction.  Sander currently does not do this, which is
      ! rather bogus...

      if (write_restrt) then
        call write_beeman_restrt(atm_cnt, t, prmtop_ititl, crd, vel)
      end if

      ! End restrt write

      do m = 1, si_cnt
        sit(m) = sit(m) + si(m)
        sit2(m) = sit2(m) + si(m) * si(m)
      end do

      ! Print step information as required...

      if (nstep .eq. 1 .or. nstep .eq. nstlim .or. mod(nstep, ntpr) .eq. 0) then
        call prntmd(nstep, t, si, fac, 7, .false.)
        if (nmropt .ne. 0) call nmrptx(6)
      end if
  
      ! Begin running_ave:

      if (ntave .gt. 0) then
        if (mod(nstep, ntave) .eq. 0) then

          ! Coming into this loop, the _tmp variables hold the values of
          ! sit, sit2 when this routine was last called (or 0.d0).  The _tmp
          ! vars are used here as scatch space and then updated with the
          ! current sit, sit2.

          do m = 1, si_cnt
            sit_tmp(m) = (sit(m) - sit_tmp(m)) / dble(ntave)
            sit2_tmp(m) = (sit2(m) - sit2_tmp(m)) / dble(ntave) - &
                          sit_tmp(m) * sit_tmp(m)
            if (sit2_tmp(m) .lt. 0.d0) sit2_tmp(m) = 0.d0
            sit2_tmp(m) = sqrt(sit2_tmp(m))
          end do

          write(mdout, '(/5x, a, i7, a, /)') ' A V E R A G E S   O V E R ', &
                                             ntave, ' S T E P S'

          call prntmd(nstep, t, sit_tmp, fac, 0, .false.)

          write(mdout, '(/5x, a, /)') ' R M S  F L U C T U A T I O N S'

          call prntmd(nstep, t, sit2_tmp, fac, 0, .false.)

          write(mdout, 542)

          sit_tmp(:) = sit(:)
          sit2_tmp(:) = sit2(:)

        end if
      end if

      ! End running_ave.

    end if

  end do

  ! Begin overall_ave:

  if (master .and. nstlim .gt. 0) then

    do m = 1, si_cnt
      sit(m) = sit(m)/dble(nstlim)
      sit2(m) = sit2(m)/dble(nstlim) - sit(m) * sit(m)
      if (sit2(m) .lt. 0.d0) sit2(m) = 0.d0
      sit2(m) =  sqrt(sit2(m))
    end do

    write(mdout, '(/5x, a, i7, a, /)') &
      ' A V E R A G E S   O V E R ', nstlim, ' S T E P S'

    call prntmd(nstlim, t, sit, fac, 0, .false.)
    if (nmropt .ne. 0) call nmrptx(6)

    write(mdout, '(/5x, a, /)') ' R M S  F L U C T U A T I O N S'

    call prntmd(nstlim, t, sit2, fac, 0, .false.)

    if (nmropt .ne. 0) then
      write(mdout, '(/,a,/)') ' NMR restraints on final step:'
      call ndvptx(crd, frc, 6)
    end if

  end if

  ! End overall_ave.

  call update_time(runmd_time)

  return

  542 format('|',79('='))

end subroutine beeman_runmd

!*******************************************************************************!
! Subroutine:  write_beeman_restrt
!
! Description: <TBS>
!
!*******************************************************************************

subroutine write_beeman_restrt(atm_cnt, sim_time, title, crd, vel)

  use file_io_dat_mod
  use file_io_mod
  use inpcrd_dat_mod
  use mdin_ctrl_dat_mod
  use pbc_mod

  ! BUGBUG - May want to implement axis flipping at some point...

  implicit none

! Formal arguments:

  integer, intent(in)                   :: atm_cnt
  double precision, intent(in)          :: sim_time
  character(len = *), intent(in)        :: title
  double precision, intent(in)          :: crd(3, *)
  double precision, intent(in)          :: vel(3, *)

! Local variables:

  integer                               :: values(8), j, n
  character(len = 12)                   :: date, time, zone
  character(len = 8)                    :: word, word1
  character(len = 16)                   :: word2

  if (ntxo .eq. 0) then
    call amopen(restrt, restrt_name, 'U', 'U', 'W')
  else
    call amopen(restrt, restrt_name, 'U', 'F', 'W')
  end if

  call date_and_time(date, time, zone, values)

  write(restrt, '(a, i2.2, a, i2.2, a, i2.2, a, i2.2, a, i2.2, a, i2.2)')  &
    '%VERSION  VERSION_STAMP = V0001.000  DATE = ', &
    values(2), '/', values(3), '/', values(1)-2000, &
    '  ', values(5), ':', values(6), ':', values(7)

  write(restrt, '(a)') '%FLAG TITLE'
  write(restrt, '(a)') '%FORMAT(a)'
  write(restrt, '(a)') title(1:len_trim(title))
  write(restrt, '(a)') '%FLAG ATOMIC_COORDS_SIMULATION_TIME'
  write(restrt, '(a)') '%FORMAT(E16.8)'
  write(restrt, '(E16.8)') sim_time

  write(restrt, '(a)') '%FLAG ATOMIC_COORDS_NUM_LIST'
  write(restrt, '(a)') '%FORMAT(i8)'
  write(restrt, '(I8)') atm_cnt
  write(word, '(I8)') atm_cnt
  word1 = adjustl(word)
  word2 = '(3,'//word1(1:len_trim(word1))//')'
  write(restrt, '(a)') '%FLAG ATOMIC_COORDS_LIST'
  write(restrt, '(a)') '%COMMENT   dimension = ' // word2
  write(restrt, '(a)') '%FORMAT(3e20.12)'
  write(restrt, '(3e20.12)')((crd(j, n), j = 1, 3), n = 1, atm_cnt)
  write(restrt, '(a)') '%FLAG ATOMIC_VELOCITIES_NUM_LIST'
  write(restrt, '(a)') '%FORMAT(i8)'
  write(restrt, '(I8)') atm_cnt
  write(restrt, '(a)') '%FLAG ATOMIC_VELOCITIES_LIST'
  write(restrt, '(a)') '%COMMENT   dimension = ' // word2
  write(restrt, '(a)') '%FORMAT(3e20.12)'
  write(restrt, '(3e20.12)')((tinker_to_sander_time_convert * &
                              vel(j, n), j = 1, 3), n = 1, atm_cnt)
  write(restrt, '(a)') '%FLAG ATOMIC_ACCELERATIONS_NUM_LIST'
  write(restrt, '(a)') '%FORMAT(i8)'
  write(restrt, '(I8)') atm_cnt
  write(restrt, '(a)') '%FLAG ATOMIC_ACCELERATIONS_LIST'
  write(restrt, '(a)') '%COMMENT   dimension = ' // word2
  write(restrt, '(a)') '%FORMAT(3e20.12)'
  write(restrt, '(3e20.12)')((tinker_to_sander_time_convert**2 * &
                              acceleration(j, n), j = 1, 3), n = 1, atm_cnt)
  write(restrt, '(a)') '%FLAG OLD_ATOMIC_ACCELERATIONS_NUM_LIST'
  write(restrt, '(a)') '%FORMAT(i8)'
  write(restrt, '(I8)') atm_cnt
  write(restrt, '(a)') '%FLAG OLD_ATOMIC_ACCELERATIONS_LIST'
  write(restrt, '(a)') '%COMMENT   dimension = ' // word2
  write(restrt, '(a)') '%FORMAT(3e20.12)'
  write(restrt, '(3e20.12)')((tinker_to_sander_time_convert**2 * &
                              old_acceleration(j, n), j = 1, 3), n = 1, atm_cnt)
  write(restrt, '(a)') '%FLAG UNIT_CELL_PARAMETERS'
  write(restrt, '(a, a)') '%COMMENT lengths a,b,c; ', &
                          'then angles alpha,beta,gamma'
  write(restrt, '(a)') '%FORMAT(3e20.12)'
  write(restrt, '(3e20.12)') pbc_box(1), pbc_box(2), pbc_box(3), &
                             pbc_alpha, pbc_beta, pbc_gamma

  close(restrt)

  return

end subroutine write_beeman_restrt

!*******************************************************************************!
! Subroutine:  beeman_vel_archive
!
! Description: <TBS>
!
!*******************************************************************************

subroutine beeman_vel_archive(atm_cnt, vel)

  use runfiles_mod
  use file_io_dat_mod

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt
  double precision, intent(in)  :: vel(3, atm_cnt)

! Local variables:

  double precision              :: scaled_vel(3, atm_cnt)

  scaled_vel(:,:) = tinker_to_sander_time_convert * vel(:,:)

  call corpac(3 * atm_cnt, scaled_vel, 1, mdvel)

  return

end subroutine beeman_vel_archive

#endif /* AMOEBA */
end module amoeba_runmd_mod
