#include "copyright.i"

!*******************************************************************************!
! Module: amoeba_induced_mod
!
! Description: <TBS>
!
!*******************************************************************************

module amoeba_induced_mod
#ifdef AMOEBA

  implicit none

  private

! Data that should be broadcast to slave processes from the master:

  integer, parameter    :: amoeba_induced_int_cnt = 2

  integer                       do_amoeba_induced_flag, polar_atm_cnt

  common / amoeba_induced_int / do_amoeba_induced_flag, polar_atm_cnt

  save  :: / amoeba_induced_int /

  double precision, save, allocatable   :: polarizability(:)

  ! Data that is allocated and initialized in final amoeba setup.  Some
  ! of this may not need to be global (BUGBUG - check it out).

  logical, save, allocatable            :: is_polarizable(:)
  double precision, save, allocatable   :: ind_dip_d(:,:)
  double precision, save, allocatable   :: ind_dip_p(:,:)

  ! Only relevant in master...

  logical, save, public                 :: print_amoeba_dip_info

  public        polarizability
  public        is_polarizable
  public        ind_dip_d
  public        ind_dip_p
  public        am_induced_zero_flag
  public        am_induced_set_user_bit
  public        init_amoeba_induced_dat
  public        amoeba_induced_dat_final_setup
  public        am_induced_eval
#ifdef MPI
  public        bcast_amoeba_induced_dat
#endif

contains

!*******************************************************************************!
! Subroutine:  am_induced_zero_flag
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_induced_zero_flag
  implicit none
  do_amoeba_induced_flag = 0
  return
end subroutine am_induced_zero_flag

!*******************************************************************************!
! Subroutine:  am_induced_set_user_bit
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_induced_set_user_bit(do_this)

  use amoeba_flags_mod
  use file_io_dat_mod
  use parallel_dat_mod

  implicit none

! Formal arguments:

  integer, intent(in) :: do_this

  if (do_this .eq. 1) then
    do_amoeba_induced_flag = ibset(do_amoeba_induced_flag, user_bit)
    print_amoeba_dip_info = .true.
  else
    do_amoeba_induced_flag = ibclr(do_amoeba_induced_flag, user_bit)
    print_amoeba_dip_info = .false.
  end if

  return

end subroutine am_induced_set_user_bit

!*******************************************************************************!
! Function:  init_amoeba_induced_dat
!
! Description: <TBS>
!
!*******************************************************************************

function init_amoeba_induced_dat(atm_cnt, num_ints, num_reals)

  use file_io_dat_mod
  use parallel_dat_mod
  use prmtop_dat_mod
  use amoeba_flags_mod

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)       :: num_ints, num_reals

! Local variables:

  integer                       :: init_amoeba_induced_dat

  init_amoeba_induced_dat = 0

  call amoeba_get_numlist('AMOEBA_POLARIZABILITY_', prmtop, polar_atm_cnt)

  if (polar_atm_cnt .le. 0) then
    do_amoeba_induced_flag = ibclr(do_amoeba_induced_flag, valid_bit)
    return
  end if

  if (polar_atm_cnt .ne. atm_cnt) then
    write(mdout, *) 'mismatch between polarizability and atm_cnt'
    call mexit(mdout, 1)
  end if

  call alloc_amoeba_induced_mem(num_ints, num_reals)

  call amoeba_read_real_list_data('AMOEBA_POLARIZABILITY_', prmtop, &
                                  1, polar_atm_cnt, polarizability)


  init_amoeba_induced_dat = 1

  do_amoeba_induced_flag = ibset(do_amoeba_induced_flag, valid_bit)

  return

end function init_amoeba_induced_dat

#ifdef MPI
!*******************************************************************************
!
! Subroutine:  bcast_amoeba_induced_dat
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine bcast_amoeba_induced_dat

  use pmemd_lib_mod
  use parallel_dat_mod
  use amoeba_flags_mod

  implicit none

  integer               :: alloc_failed
  integer               :: num_ints, num_reals  ! returned values discarded

  call mpi_bcast(do_amoeba_induced_flag, amoeba_induced_int_cnt, &
                 mpi_integer, 0, mpi_comm_world, err_code_mpi)

  if (polar_atm_cnt .le. 0) return

  if (.not. master) then
    num_ints = 0
    num_reals = 0
    call alloc_amoeba_induced_mem(num_ints, num_reals)
  end if

  call mpi_bcast(polarizability, polar_atm_cnt, mpi_double_precision, 0, &
                 mpi_comm_world, err_code_mpi)

  return

end subroutine bcast_amoeba_induced_dat
#endif /* MPI */

!*******************************************************************************
!
! Subroutine:  alloc_amoeba_induced_mem
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine alloc_amoeba_induced_mem(num_ints, num_reals)

  use pmemd_lib_mod
  use parallel_dat_mod
  use amoeba_flags_mod

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)       :: num_ints, num_reals     

! Local variables:

  integer                       :: alloc_failed

  allocate(polarizability(polar_atm_cnt), &
           stat = alloc_failed)

  if (alloc_failed .ne. 0) call setup_alloc_error

  num_reals = num_reals + size(polarizability)

  polarizability(:) = 0.d0

  return

end subroutine alloc_amoeba_induced_mem

!*******************************************************************************
!
! Subroutine:  amoeba_induced_dat_final_setup
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine amoeba_induced_dat_final_setup(num_ints, num_reals)

  use pmemd_lib_mod
  use parallel_dat_mod
  use prmtop_dat_mod
  use amoeba_flags_mod

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)       :: num_ints, num_reals     

! Local variables:

  integer                       :: alloc_failed
  integer                       :: n
  double precision, parameter   :: polmin = 1.d-12

  ! We allocate and initialize these data structures even in induction is
  ! turned off because there is code that references these data structures
  ! that will blow up.

  if (polar_atm_cnt .le. 0) return

  allocate(is_polarizable(polar_atm_cnt), &
           ind_dip_d(3, polar_atm_cnt), &
           ind_dip_p(3, polar_atm_cnt), &
           stat = alloc_failed)

  if (alloc_failed .ne. 0) call setup_alloc_error

  num_reals = num_reals + size(ind_dip_d) + &
                          size(ind_dip_p)

  num_ints = num_ints + size(is_polarizable)   ! count logicals as integers...

  ! The data previously in the sq_polinv array is now stored in atm_qterm()
  ! over in prmtop_dat_mod.  This is the easiest way to switch between using
  ! a field in the img type for something useful in both amber pme and amoeba.

  do n = 1, polar_atm_cnt
    if (polarizability(n) .gt. polmin) then
      is_polarizable(n) = .true.
      atm_qterm(n) = 1.d0 / sqrt(polarizability(n))
    else
      is_polarizable(n) = .false.
      atm_qterm(n) = 1.d0 / sqrt(polmin)
    end if
    ind_dip_d(:, n) = 0.d0
    ind_dip_p(:, n) = 0.d0
  end do

  return

end subroutine amoeba_induced_dat_final_setup

!*******************************************************************************!
! Subroutine:  am_induced_eval
!
! Description: <TBS>
!
!*******************************************************************************

#ifdef MPI
! MPI IMPLEMENTATION
subroutine am_induced_eval(atm_cnt, crd, diprms, dipiter, num_adjust_list)

  use mdin_amoeba_dat_mod, only : sor_coefficient, dipole_scf_tol, &
                                  dipole_scf_iter_max, amoeba_verbose

  use amoeba_flags_mod
  use file_io_dat_mod
  use parallel_dat_mod
  use timers_mod

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt
  double precision, intent(in)  :: crd(3, *)
  double precision, intent(out) :: diprms
  double precision, intent(out) :: dipiter
  integer, intent(in)           :: num_adjust_list

! Local variables:

  integer                       :: iter
  integer, save                 :: done
  double precision              :: rms1, rms2, rms, oldrms
  double precision              :: dip_field_d(3, atm_cnt)
  double precision              :: dip_field_p(3, atm_cnt)
  double precision              :: dip_d_perm(3, atm_cnt)
  double precision              :: dip_p_perm(3, atm_cnt)
  double precision              :: old_dip_d(3, atm_cnt)
  double precision              :: old_dip_p(3, atm_cnt)
  double precision              :: reduced_dip_d(3, atm_cnt)
  double precision              :: reduced_dip_p(3, atm_cnt)
  double precision              :: adj_dip_dip_tensor(6, num_adjust_list)


  diprms = 0.d0
  dipiter = 0.d0

  if (do_amoeba_induced_flag .ne. proceed) return

  dip_field_d(:,:) = 0.d0
  dip_field_p(:,:) = 0.d0

  call am_nonbond_perm_fields(atm_cnt, is_polarizable, crd, &
                              dip_field_d, dip_field_p, adj_dip_dip_tensor)

  call update_time(nonbond_time)

  call mpi_allreduce(dip_field_d, dbl_mpi_recv_buf, 3 * atm_cnt, &
                     mpi_double_precision, mpi_sum, mpi_comm_world, &
                     err_code_mpi)

  call array_copy(reduced_dip_d, dbl_mpi_recv_buf, 3 * atm_cnt)

  call mpi_allreduce(dip_field_p, dbl_mpi_recv_buf, 3 * atm_cnt, &
                     mpi_double_precision, mpi_sum, mpi_comm_world, &
                     err_code_mpi)

  call array_copy(reduced_dip_p, dbl_mpi_recv_buf, 3 * atm_cnt)


  call update_time(fcve_dist_time)
  call zero_pme_time()

  call am_induced_fields_to_ind_dips(atm_cnt, is_polarizable, &
                                     dip_field_d, dip_field_p, &
                                     polarizability, ind_dip_d, ind_dip_p)

  ! Do an all reduce on the ind_dip_[dp] arrays to get input for
  ! am_nonbond_dip_dip_fields

  call update_time(nonbond_time)

  call mpi_allreduce(ind_dip_d, dbl_mpi_recv_buf, 3 * atm_cnt, &
                     mpi_double_precision, mpi_sum, mpi_comm_world, &
                     err_code_mpi)

  call array_copy(reduced_dip_d, dbl_mpi_recv_buf, 3 * atm_cnt)

  call mpi_allreduce(ind_dip_p, dbl_mpi_recv_buf, 3 * atm_cnt, &
                     mpi_double_precision, mpi_sum, mpi_comm_world, &
                     err_code_mpi)

  call array_copy(reduced_dip_p, dbl_mpi_recv_buf, 3 * atm_cnt)

  ! Save the dips due to permanent fields

  dip_d_perm(:,1:atm_cnt) = reduced_dip_d(:,1:atm_cnt)
  dip_p_perm(:,1:atm_cnt) = reduced_dip_p(:,1:atm_cnt)

  call update_time(fcve_dist_time)
  call zero_pme_time()

  iter = 0
  rms = 1.d0

  done = 0

  do 

    ! copy dipoles to old

    old_dip_d(:,1:atm_cnt) = reduced_dip_d(:,1:atm_cnt)
    old_dip_p(:,1:atm_cnt) = reduced_dip_p(:,1:atm_cnt)

    call am_nonbond_dip_dip_fields(atm_cnt, crd, &
                                   reduced_dip_d, reduced_dip_p,  &
                                   adj_dip_dip_tensor, dip_field_d, dip_field_p)

    ! get dipoles due to dipole fields

    call am_induced_fields_to_ind_dips(atm_cnt, is_polarizable, &
                                       dip_field_d, dip_field_p, &
                                       polarizability, ind_dip_d, ind_dip_p)

    call update_time(nonbond_time)

    ! Get the next reduced set of values for ind_dip_[dp]:

    call mpi_allreduce(ind_dip_d, dbl_mpi_recv_buf, 3 * atm_cnt, &
                       mpi_double_precision, mpi_sum, mpi_comm_world, &
                       err_code_mpi)

    call array_copy(reduced_dip_d, dbl_mpi_recv_buf, 3 * atm_cnt)


    call mpi_allreduce(ind_dip_p, dbl_mpi_recv_buf, 3 * atm_cnt, &
                       mpi_double_precision, mpi_sum, mpi_comm_world, &
                       err_code_mpi)

    call array_copy(reduced_dip_p, dbl_mpi_recv_buf, 3 * atm_cnt)

    call update_time(fcve_dist_time)
    call zero_pme_time()

    ! add dipoles due to permfields

    call array_add(reduced_dip_d, dip_d_perm, 3 * atm_cnt)
    call array_add(reduced_dip_p, dip_p_perm, 3 * atm_cnt)

    call am_induced_sor_update(atm_cnt, is_polarizable, sor_coefficient, &
                               reduced_dip_d, reduced_dip_p,  &
                               old_dip_d, old_dip_p)

    ! In theory, everyone would get the same result if they executed the
    ! following code.  In reality, there may be rounding errors in mpi
    ! dependent on operation order, so we just have the master do the next
    ! step.  Note however, that we have already gotten the result to all
    ! processes once we converge...

    if (master) then

      oldrms = rms

      call am_induced_rms_diff(atm_cnt, is_polarizable, rms1, rms2, &
                               reduced_dip_d, reduced_dip_p,  &
                               old_dip_d, old_dip_p)
      rms = max(rms1, rms2)
      iter = iter + 1

      call update_pme_time(pme_misc_timer)
      call update_time(nonbond_time)
      if (rms .lt. dipole_scf_tol) then
        done = 1
        call mpi_bcast(done, 1, mpi_integer, 0, mpi_comm_world, err_code_mpi)
      else if (rms .gt. oldrms) then
        write(mdout,  * ) 'dipoles failed to converge: rms increasing!'
        done = -1 ! Error flag...
        call mpi_bcast(done, 1, mpi_integer, 0, mpi_comm_world, err_code_mpi)
        call mexit(mdout, 1)
      else if (iter .gt. dipole_scf_iter_max) then
        done = -1 ! Error flag...
        write(mdout, *) 'dipoles failed to converge: iter max exceeded!'
        call mpi_bcast(done, 1, mpi_integer, 0, mpi_comm_world, err_code_mpi)
        call mexit(mdout, 1)
      else
        call mpi_bcast(done, 1, mpi_integer, 0, mpi_comm_world, err_code_mpi)
      end if
    else
      call update_pme_time(pme_misc_timer)
      call update_time(nonbond_time)
      call mpi_bcast(done, 1, mpi_integer, 0, mpi_comm_world, err_code_mpi)
    end if
    call update_time(fcve_dist_time)
    call zero_pme_time()


    if (done .gt. 0) then
      exit
    else if (done .lt. 0) then
      call mexit(0, 1)
    end if

  end do

  ind_dip_d(:,:) = reduced_dip_d(:,:)
  ind_dip_p(:,:) = reduced_dip_p(:,:)

  if (master .and. amoeba_verbose .ne. 0) &
     write(mdout, '(a, i5, 1x, e12.4)') 'out of loop, iter,rms = ', iter, rms

  if (master) then
    diprms = rms
    dipiter = iter
  end if

  call update_pme_time(pme_misc_timer)

  return

end subroutine am_induced_eval
#else
! UNIPROCESSOR IMPLEMENTATION
subroutine am_induced_eval(atm_cnt, crd, diprms, dipiter, num_adjust_list)

  use mdin_amoeba_dat_mod, only : sor_coefficient, dipole_scf_tol, &
                                  dipole_scf_iter_max, amoeba_verbose

  use amoeba_flags_mod
  use file_io_dat_mod
  use parallel_dat_mod
  use timers_mod

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt
  double precision, intent(in)  :: crd(3, *)
  double precision, intent(out) :: diprms
  double precision, intent(out) :: dipiter
  integer, intent(in)           :: num_adjust_list

! Local variables:

  integer                       :: iter
  double precision              :: rms1, rms2, oldrms
  double precision, save        :: rms
  double precision              :: dip_field_d(3, atm_cnt)
  double precision              :: dip_field_p(3, atm_cnt)
  double precision              :: dip_d_perm(3, atm_cnt)
  double precision              :: dip_p_perm(3, atm_cnt)
  double precision              :: old_dip_d(3, atm_cnt)
  double precision              :: old_dip_p(3, atm_cnt)
  double precision              :: adj_dip_dip_tensor(6, num_adjust_list)

  if (do_amoeba_induced_flag .ne. proceed) return

  dip_field_d(:,:) = 0.d0
  dip_field_p(:,:) = 0.d0

  call am_nonbond_perm_fields(atm_cnt, is_polarizable, crd, &
                              dip_field_d, dip_field_p, adj_dip_dip_tensor)

  call am_induced_fields_to_ind_dips(atm_cnt, is_polarizable, &
                                     dip_field_d, dip_field_p, &
                                     polarizability, ind_dip_d, ind_dip_p)

  ! Save the dips due to permanent fields

  dip_d_perm(:,1:atm_cnt) = ind_dip_d(:,1:atm_cnt)
  dip_p_perm(:,1:atm_cnt) = ind_dip_p(:,1:atm_cnt)

  iter = 0
  rms = 1.d0

  call update_pme_time(pme_misc_timer)

  do 

    ! copy dipoles to old

    old_dip_d(:,1:atm_cnt) = ind_dip_d(:,1:atm_cnt)
    old_dip_p(:,1:atm_cnt) = ind_dip_p(:,1:atm_cnt)

    call am_nonbond_dip_dip_fields(atm_cnt, crd, ind_dip_d, ind_dip_p,  &
                                   adj_dip_dip_tensor, dip_field_d, dip_field_p)

    ! get dipoles due to dipole fields

    call am_induced_fields_to_ind_dips(atm_cnt, is_polarizable, &
                                       dip_field_d, dip_field_p, &
                                       polarizability, ind_dip_d, ind_dip_p)

    ! add dipoles due to permfields

    call array_add(ind_dip_d, dip_d_perm, 3 * atm_cnt)
    call array_add(ind_dip_p, dip_p_perm, 3 * atm_cnt)

    call am_induced_sor_update(atm_cnt, is_polarizable, sor_coefficient, &
                               ind_dip_d, ind_dip_p,  old_dip_d, old_dip_p)

    call am_induced_rms_diff(atm_cnt, is_polarizable, rms1, rms2, &
                             ind_dip_d, ind_dip_p,  old_dip_d, old_dip_p)

    oldrms = rms
    rms = max(rms1, rms2)
    iter = iter + 1

    if (rms .lt. dipole_scf_tol) exit

    if (rms .gt. oldrms) then
        write(mdout,  * ) 'dipoles failed to converge: rms increasing!'
        call mexit(mdout, 1)
    end if

    if (iter .gt. dipole_scf_iter_max) then
        write(mdout, *) 'dipoles failed to converge: iter max exceeded!'
        call mexit(mdout, 1)
    end if

  end do

  call update_pme_time(pme_misc_timer)

  if (master .and. amoeba_verbose .ne. 0) &
     write(mdout, '(a, i5, 1x, e12.4)') 'out of loop, iter,rms = ', iter, rms

  dipiter = iter
  diprms = rms

  return

end subroutine am_induced_eval
#endif /* not MPI */

#endif /* AMOEBA */
end module amoeba_induced_mod

#ifdef AMOEBA
!*******************************************************************************!
! Subroutine:  am_induced_add_cart_to_dip
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_induced_add_cart_to_dip(atm_cnt, is_polarizable, &
                                      cart_dipole_field, &
                                      dip_field_d, dip_field_p)

  use timers_mod

  implicit none

! Formal arguments:

  integer, intent(in)                   :: atm_cnt
  logical, intent(in)                   :: is_polarizable(*)
  double precision, intent(in)          :: cart_dipole_field(3, *)
  double precision, intent(in out)      :: dip_field_d(3, *)
  double precision, intent(in out)      :: dip_field_p(3, *)

! Local variables:

  integer                               :: n

  ! NOTE - This is intentionally over entire atom range at the moment.

  do n = 1, atm_cnt
    if (is_polarizable(n)) then
      dip_field_d(:, n) = dip_field_d(:, n) + cart_dipole_field(:, n)
      dip_field_p(:, n) = dip_field_p(:, n) + cart_dipole_field(:, n)
    end if
  end do

  call update_pme_time(pme_misc_timer)

  return

end subroutine am_induced_add_cart_to_dip

!*******************************************************************************!
! Subroutine:  am_induced_fields_to_ind_dips
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_induced_fields_to_ind_dips(atm_cnt, is_polarizable, &
                                         dip_field_d, dip_field_p, &
                                         polarizability, &
                                         ind_dip_d, ind_dip_p)

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt
  logical, intent(in)           :: is_polarizable(*)
  double precision, intent(in)  :: dip_field_d(3, *)
  double precision, intent(in)  :: dip_field_p(3, *)
  double precision, intent(in)  :: polarizability(*)
  double precision, intent(out) :: ind_dip_d(3, *)
  double precision, intent(out) :: ind_dip_p(3, *)

! Local variables:

  integer                       :: n
  double precision              :: pol

  ! NOTE - This is intentionally over entire atom range at the moment.

  do n = 1, atm_cnt

    if (is_polarizable(n)) then
      pol = polarizability(n)
      ! minus sign since our field is actually gradient of potential
      ind_dip_d(:, n) = -pol * dip_field_d(:, n)
      ind_dip_p(:, n) = -pol * dip_field_p(:, n)
    else
      ind_dip_d(:, n) = 0.d0
      ind_dip_p(:, n) = 0.d0
    end if
  end do

  return

end subroutine am_induced_fields_to_ind_dips

!*******************************************************************************!
! Subroutine:  am_induced_sor_update
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_induced_sor_update(atm_cnt, is_polarizable, sor_coeff, &
                                 ind_dip_d, ind_dip_p, old_dip_d, old_dip_p)

  implicit none

! Formal arguments:

  integer, intent(in)                   :: atm_cnt
  logical, intent(in)                   :: is_polarizable(*)
  double precision, intent(in)          :: sor_coeff
  double precision, intent(in out)      :: ind_dip_d(3, *)
  double precision, intent(in out)      :: ind_dip_p(3, *)
  double precision, intent(in)          :: old_dip_d(3, *)
  double precision, intent(in)          :: old_dip_p(3, *)

! Local variables:

  double precision                      :: c_sor
  integer                               :: i, n

  ! NOTE - This is intentionally over entire atom range at the moment.

  c_sor = 1.d0 - sor_coeff

  do n = 1, atm_cnt
    if (is_polarizable(n)) then
      ind_dip_d(:, n) = c_sor * old_dip_d(:, n) + sor_coeff * ind_dip_d(:, n)
      ind_dip_p(:, n) = c_sor * old_dip_p(:, n) + sor_coeff * ind_dip_p(:, n)
    end if
  end do

  return

end subroutine am_induced_sor_update

!*******************************************************************************!
! Subroutine:  am_induced_rms_diff
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_induced_rms_diff(atm_cnt, is_polarizable, rms1, rms2, &
                               ind_dip_d, ind_dip_p, old_dip_d, old_dip_p)

  use file_io_dat_mod
  use parallel_dat_mod

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt
  logical, intent(in)           :: is_polarizable(*)
  double precision, intent(out) :: rms1
  double precision, intent(out) :: rms2
  double precision, intent(in)  :: ind_dip_d(3, *)
  double precision, intent(in)  :: ind_dip_p(3, *)
  double precision, intent(in)  :: old_dip_d(3, *)
  double precision, intent(in)  :: old_dip_p(3, *)

! Local variables:

  double precision              :: debye
  integer                       :: n, num

  debye = 4.8033324d0
  num = 0
  rms1 = 0.d0
  rms2 = 0.d0

  do n = 1, atm_cnt

    if (is_polarizable(n)) then
      num = num + 1

      rms1 = rms1 + (ind_dip_d(1, n)-old_dip_d(1, n))**2 + &
                    (ind_dip_d(2, n)-old_dip_d(2, n))**2 + &
                    (ind_dip_d(3, n)-old_dip_d(3, n))**2

      rms2 = rms2 + (ind_dip_p(1, n)-old_dip_p(1, n))**2 + &
                    (ind_dip_p(2, n)-old_dip_p(2, n))**2 + &
                    (ind_dip_p(3, n)-old_dip_p(3, n))**2
    end if
  end do

  if (num .eq. 0) then
    write(mdout, *)'am_induced_rms_diff: num polarizable = 0!!'
    call mexit(mdout, 1)
  end if

  rms1 = debye * sqrt(rms1 / num)
  rms2 = debye * sqrt(rms2 / num)

  return

end subroutine am_induced_rms_diff
#endif /* AMOEBA */
