#include "copyright.i"

!*******************************************************************************
!
! Module:  img_mod
!
! Description: <TBS>
!              
!*******************************************************************************

module img_mod

  use gbl_datatypes_mod

  implicit none

  double precision, save, allocatable           :: gbl_img_crd(:,:)
  ! qterm is atom charge under pme or gb and sq_polinv under amoeba:
  double precision, save, allocatable           :: gbl_img_qterm(:)
  ! gbl_img_iac vdw info only applies to "regular" pme, not amoeba!
  integer, save, allocatable                    :: gbl_img_iac(:)

  integer, save, allocatable                    :: gbl_atm_img_map(:)
  integer, save, allocatable                    :: gbl_img_atm_map(:)
#ifdef MPI
  ! used images are "1" in the map; unused images are "0"
  integer(byte), save, allocatable              :: gbl_used_img_map(:)
#endif /* MPI */
  integer, save, allocatable                    :: gbl_excl_img_flags(:)
  double precision, save                        :: gbl_tranvec(1:3,0:17)

  ! my_img_* covers the range of images you "own"; ie. forces are accumulated
  ! in the local process for those images.  These are assigned WITHOUT wrapping.
  ! used_img_* covers the range of images you "use"; ie. you may need
  ! coordinate information and will report nonbonded force information to the
  ! owner for some images in this range. This range may wrap through natom to 1.

  integer, save         :: my_img_lo, my_img_hi
  integer, save         :: used_img_lo, used_img_hi

#ifdef MPI
  logical, save         :: used_img_range_wraps ! from lo through natom to hi
#endif

contains

!*******************************************************************************
!
! Subroutine:  alloc_img_mem
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine alloc_img_mem(atm_cnt, num_ints, num_reals)

  use pmemd_lib_mod

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer                       :: atm_cnt
  integer, intent(in out)       :: num_ints, num_reals

! Local variables:

  integer               :: alloc_failed

#ifndef MPI
  ! In an MPI version, this stuff will be set elsewhere.

  my_img_lo = 1
  my_img_hi = atm_cnt
  used_img_lo = 1
  used_img_hi = atm_cnt
#endif

  allocate(gbl_atm_img_map(atm_cnt), &
           gbl_img_atm_map(atm_cnt), &
#ifdef MPI
           gbl_used_img_map(atm_cnt), &
#endif
           gbl_img_crd(3, atm_cnt), &
           gbl_img_qterm(atm_cnt), &
           gbl_img_iac(atm_cnt), &
           gbl_excl_img_flags(atm_cnt), &
           stat = alloc_failed)

  if (alloc_failed .ne. 0) call setup_alloc_error

  num_reals = num_reals + size(gbl_img_crd) + &
                          size(gbl_img_qterm)

  num_ints = num_ints + size(gbl_atm_img_map) + &
                        size(gbl_img_atm_map) + &
#ifdef MPI
                        size(gbl_used_img_map) / 4 + &
#endif
                        size(gbl_img_iac) + &
                        size(gbl_excl_img_flags)

  gbl_img_crd(:,:) = 0.d0
  gbl_img_qterm(:) = 0.d0
  gbl_img_iac(:) = 0

  gbl_atm_img_map(:) = 0
  gbl_img_atm_map(:) = 0
#ifdef MPI
  gbl_used_img_map(:) = 0
#endif
  gbl_excl_img_flags(:) = 0

  return

end subroutine alloc_img_mem

#ifdef MPI
!*******************************************************************************
!
! Subroutine:  bcast_img_dat
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine bcast_img_dat(atm_cnt)

  use parallel_dat_mod

  implicit none

! Formal arguments:

  integer       :: atm_cnt

! Local variables:

  integer       :: num_ints, num_reals  ! returned values discarded

  ! Nothing to broadcast.  We just allocate storage in the non-master nodes.

  if (.not. master) then
    num_ints = 0
    num_reals = 0
    call alloc_img_mem(atm_cnt, num_ints, num_reals)
  end if

  ! The allocated data is not initialized from the master node.

  return

end subroutine bcast_img_dat
#endif

!*******************************************************************************
!
! Subroutine:  fill_tranvec
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine fill_tranvec(tranvec)

  use pbc_mod

  implicit none

  double precision      :: tranvec(3, 18) ! this is (1:3,0:17) externally.
  integer               :: iv, i0, i1, i2, i3

! This works for both orthogonal and nonorthogonal unit cells.

  iv = 0

  do i3 = -1, 0
    do i2 = -1, 1
      do i1 = -1, 1
        iv = iv + 1
        do i0 = 1, 3
          tranvec(i0, iv) = i1 * ucell(i0, 1) + &
                            i2 * ucell(i0, 2) + &
                            i3 * ucell(i0, 3)
        end do
      end do
    end do
  end do

  return

end subroutine fill_tranvec


#ifdef MPI
!*******************************************************************************
!
! Subroutine:  find_img_range
!
! Description:  Find the range of images this process uses.
!
!*******************************************************************************

subroutine find_img_range(img_cnt, used_img_map)

  use parallel_dat_mod

  implicit none

! Formal arguments:

  integer               :: img_cnt
  integer(byte)         :: used_img_map(img_cnt)

! Local variables:

  integer               :: img_i
  integer               :: run_lo, run_hi, run_cnt
  integer               :: lo_run_lo, lo_run_hi, lo_run_cnt
  integer               :: hi_run_lo, hi_run_hi, hi_run_cnt
  logical               :: in_run

  run_cnt = 0
  lo_run_cnt = 0

  in_run = .false.
  
! Find the longest unused run of images with id's below my_img_lo:

  do img_i = 1, my_img_lo

    if (used_img_map(img_i) .eq. 0) then   ! img not used
      if (in_run) then
        run_hi = img_i
      else                
        run_lo = img_i
        run_hi = img_i
        in_run = .true.
      end if
    else                                  ! img used
      if (in_run) then
        run_cnt = run_hi - run_lo + 1
        if (run_cnt .gt. lo_run_cnt) then
          lo_run_cnt = run_cnt
          lo_run_lo = run_lo
          lo_run_hi = run_hi
        end if
        in_run = .false.
      end if
    end if

  end do

  if (in_run) then
    run_cnt = run_hi - run_lo + 1
    if (run_cnt .gt. lo_run_cnt) then
      lo_run_cnt = run_cnt
      lo_run_lo = run_lo
      lo_run_hi = run_hi
    end if
  end if
  
  run_cnt = 0
  hi_run_cnt = 0

  in_run = .false.
  
! Find the longest unused run of images with id's above my_img_hi:

  do img_i = my_img_hi + 1, img_cnt

    if (used_img_map(img_i) .eq. 0) then   ! img not used
      if (in_run) then
        run_hi = img_i
      else                
        run_lo = img_i
        run_hi = img_i
        in_run = .true.
      end if
    else                                  ! img used
      if (in_run) then
        run_cnt = run_hi - run_lo + 1
        if (run_cnt .gt. hi_run_cnt) then
          hi_run_cnt = run_cnt
          hi_run_lo = run_lo
          hi_run_hi = run_hi
        end if
        in_run = .false.
      end if
    end if

  end do

  if (in_run) then
    run_cnt = run_hi - run_lo + 1
    if (run_cnt .gt. hi_run_cnt) then
      hi_run_cnt = run_cnt
      hi_run_lo = run_lo
      hi_run_hi = run_hi
    end if
  end if

! Now check and see if the two ranges are joined.  If so, we can combine them.
! The algorithm used in this subroutine could miss an opportunity to join
! ranges (due to finding a larger lo or hi subrange away from the boundary), but
! this is fairly unlikely, will not affect correctness of results, and will
! probably have a pretty small impact on performance (it will cause the used
! range to be slightly larger than it actually is)

  if (lo_run_cnt + hi_run_cnt .eq. 0) then      ! highly unlikely, but...

    used_img_lo = 1
    used_img_hi = img_cnt
    used_img_range_wraps = .false.

  else if (lo_run_cnt .gt. 0 .and. &
           hi_run_cnt .gt. 0 .and. &
           lo_run_lo .eq. 1 .and. &
           hi_run_hi .eq. img_cnt) then         ! unused ranges fuse...

    used_img_lo = lo_run_hi + 1
    used_img_hi = hi_run_lo - 1
    used_img_range_wraps = .false.

  else if (lo_run_cnt .ge. hi_run_cnt) then     ! biggest unused range lo

    used_img_lo = lo_run_hi + 1
    used_img_hi = lo_run_lo - 1

    if (used_img_hi .le. 0) then
      used_img_hi = img_cnt
      used_img_range_wraps = .false.
    else
      used_img_range_wraps = .true.
    end if

  else
   
    used_img_lo = hi_run_hi + 1
    used_img_hi = hi_run_lo - 1

    if (used_img_lo .gt. img_cnt) then
      used_img_lo = 1
      used_img_range_wraps = .false.
    else
      used_img_range_wraps = .true.
    end if

  end if

! BEGIN DBG
! write(0,*)'DBG: task, used_img_lo,hi=', mytaskid, used_img_lo, used_img_hi
! write(0,*)'DBG: task, used_img_range_wraps=', mytaskid, used_img_range_wraps
! END DBG

  return

end subroutine find_img_range
#endif

#ifdef MPI
!*******************************************************************************
!
! Subroutine:   mark_owned_imgs_as_used
!
! Description:  Per name...
!
!*******************************************************************************

subroutine mark_owned_imgs_as_used

  implicit none

  gbl_used_img_map(my_img_lo:my_img_hi) = 1
    
  return

end subroutine mark_owned_imgs_as_used
#endif /* MPI */

#ifdef MPI
!*******************************************************************************
!
! Subroutine:   mark_unowned_imgs_as_unused
!
! Description:  Per name...
!
!*******************************************************************************

subroutine mark_unowned_imgs_as_unused(img_cnt)

  implicit none

! Formal arguments:

  integer               :: img_cnt

  if (my_img_lo .ne. 1) gbl_used_img_map(1:my_img_lo - 1) = 0
  if (my_img_hi .ne. img_cnt) gbl_used_img_map(my_img_hi + 1:img_cnt) = 0
    
  return

end subroutine mark_unowned_imgs_as_unused
#endif /* MPI */

#ifdef AMOEBA
#ifdef MPI
!*******************************************************************************
!
! Subroutine:   mark_all_imgs_as_used
!
! Description:  Per name...  This is needed currently for the amoeba master
!               process, which does all bonded processing.
!
!*******************************************************************************

subroutine mark_all_imgs_as_used

  implicit none

  gbl_used_img_map(:) = 1
    
  return

end subroutine mark_all_imgs_as_used
#endif /* MPI */
#endif /* AMOEBA */

end module img_mod
