#include "copyright.i"

!*******************************************************************************!
! Module: amoeba_interface_mod
!
! Description: <TBS>
!
!*******************************************************************************

module amoeba_interface_mod
#ifdef AMOEBA

  implicit none

  private

  public        init_amoeba_dat
  public        amoeba_dat_final_setup
  public        am_val_eval
  public        am_nonbond_eval
#ifdef MPI
  public        bcast_amoeba_dat
#endif

contains

!*******************************************************************************!
! Subroutine:  init_amoeba_nonbond_dat
!
! Description: <TBS>
!
!*******************************************************************************

subroutine init_amoeba_nonbond_dat(atm_cnt, num_ints, num_reals)

  use amoeba_multipoles_mod, only : init_amoeba_multipoles_dat
  use amoeba_adjust_mod, only : init_amoeba_adjust_dat
  use amoeba_vdw_mod, only : init_amoeba_vdw_dat
  use amoeba_induced_mod, only : init_amoeba_induced_dat

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)       :: num_ints, num_reals

! Local variables:

  integer               :: mpole_valid, adjust_valid, vdw_valid, polar_valid
  
  mpole_valid = init_amoeba_multipoles_dat(atm_cnt, num_ints, num_reals)
  adjust_valid = init_amoeba_adjust_dat(num_ints, num_reals)
  vdw_valid = init_amoeba_vdw_dat(num_ints, num_reals)
  polar_valid = init_amoeba_induced_dat(atm_cnt, num_ints, num_reals)

  return

end subroutine init_amoeba_nonbond_dat

!*******************************************************************************!
! Subroutine:  amoeba_nonbond_dat_final_setup
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_nonbond_dat_final_setup(num_ints, num_reals)

  use amoeba_vdw_mod, only : amoeba_vdw_dat_final_setup
  use amoeba_induced_mod, only : amoeba_induced_dat_final_setup
  use amoeba_recip_mod, only : amoeba_recip_final_setup

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)       :: num_ints, num_reals

! Local variables:

  call amoeba_vdw_dat_final_setup
  call amoeba_induced_dat_final_setup(num_ints, num_reals)
  call amoeba_recip_final_setup

  return

end subroutine amoeba_nonbond_dat_final_setup

!*******************************************************************************!
! Subroutine:  init_amoeba_valence_dat
!
! Description: <TBS>
!
!*******************************************************************************

subroutine init_amoeba_valence_dat(num_ints, num_reals)

  use amoeba_bonds_mod, only : init_amoeba_bonds_dat
  use amoeba_ureyb_mod, only : init_amoeba_ureyb_dat
  use amoeba_reg_angles_mod, only : init_amoeba_reg_angles_dat
  use amoeba_trig_angles_mod, only : init_amoeba_trig_angles_dat
  use amoeba_opbend_angles_mod, only : init_amoeba_opbend_angles_dat
  use amoeba_torsions_mod, only : init_amoeba_torsions_dat
  use amoeba_stretch_torsions_mod, only : init_amoeba_stretch_torsions_dat
  use amoeba_pitorsions_mod, only : init_amoeba_pitorsions_dat
  use amoeba_stretch_bend_mod, only : init_amoeba_stretch_bend_dat
  use amoeba_torsion_torsion_mod, only : init_amoeba_tor_tor_dat

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)      :: num_ints, num_reals

! Local variables:

  integer                       :: bonds_valid
  integer                       :: ureyb_valid
  integer                       :: angles_valid
  integer                       :: trig_angles_valid
  integer                       :: opbend_valid
  integer                       :: torsions_valid
  integer                       :: pitorsions_valid
  integer                       :: stretchbend_valid
  integer                       :: torsion_torsion_valid
  integer                       :: stretch_torsion_valid

  bonds_valid = init_amoeba_bonds_dat(num_ints, num_reals)
  ureyb_valid = init_amoeba_ureyb_dat(num_ints, num_reals)
  angles_valid = init_amoeba_reg_angles_dat(num_ints, num_reals)
  trig_angles_valid = init_amoeba_trig_angles_dat(num_ints, num_reals)
  opbend_valid = init_amoeba_opbend_angles_dat(num_ints, num_reals)
  torsions_valid = init_amoeba_torsions_dat(num_ints, num_reals)
  stretch_torsion_valid = init_amoeba_stretch_torsions_dat(num_ints, num_reals)
  pitorsions_valid = init_amoeba_pitorsions_dat(num_ints, num_reals)
  stretchbend_valid = init_amoeba_stretch_bend_dat(num_ints, num_reals)
  torsion_torsion_valid = init_amoeba_tor_tor_dat(num_ints, num_reals)

  return

end subroutine init_amoeba_valence_dat

!*******************************************************************************!
! Subroutine:  amoeba_valence_dat_final_setup
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_valence_dat_final_setup(num_ints, num_reals)

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)       :: num_ints, num_reals

! Local variables:

  ! Currently nothing needed...

  return

end subroutine amoeba_valence_dat_final_setup

!*******************************************************************************!
! Subroutine:  init_amoeba_dat
!
! Description: <TBS>
!
!*******************************************************************************

subroutine init_amoeba_dat(atm_cnt, num_ints, num_reals)

  use file_io_dat_mod
  use file_io_mod
  use nextprmtop_section_mod

  implicit none

! Formal arguments:

  integer, intent(in)          :: atm_cnt

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)      :: num_ints, num_reals

  call amopen(prmtop, prmtop_name, 'O', 'F', 'R')
  call nxtsec_reset()   ! insurance; the nextprmtop stuff caches the fileptr.

  call init_amoeba_valence_dat(num_ints, num_reals)
  call init_amoeba_nonbond_dat(atm_cnt, num_ints, num_reals)

  close(unit = prmtop)

  return

end subroutine init_amoeba_dat

!*******************************************************************************!
! Subroutine:  amoeba_dat_final_setup
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_dat_final_setup(num_ints, num_reals)

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer, intent(in out)      :: num_ints, num_reals

  call amoeba_valence_dat_final_setup(num_ints, num_reals)
  call amoeba_nonbond_dat_final_setup(num_ints, num_reals)

  return

end subroutine amoeba_dat_final_setup

#ifdef MPI
!*******************************************************************************
!
! Subroutine:  bcast_amoeba_dat
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine bcast_amoeba_dat

  use parallel_dat_mod

  implicit none

  call bcast_amoeba_valence_dat
  call bcast_amoeba_nonbond_dat

  return

end subroutine bcast_amoeba_dat
#endif

#ifdef MPI
!*******************************************************************************
!
! Subroutine:  bcast_amoeba_valence_dat
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine bcast_amoeba_valence_dat

  use pmemd_lib_mod
  use parallel_dat_mod
  use amoeba_bonds_mod, only : bcast_amoeba_bonds_dat
  use amoeba_ureyb_mod, only : bcast_amoeba_ureyb_dat
  use amoeba_reg_angles_mod, only : bcast_amoeba_reg_angles_dat
  use amoeba_trig_angles_mod, only : bcast_amoeba_trig_angles_dat
  use amoeba_opbend_angles_mod, only : bcast_amoeba_opbend_angles_dat
  use amoeba_torsions_mod, only : bcast_amoeba_torsions_dat
  use amoeba_stretch_torsions_mod, only : bcast_amoeba_stretch_torsions_dat
  use amoeba_pitorsions_mod, only : bcast_amoeba_pitorsions_dat
  use amoeba_stretch_bend_mod, only : bcast_amoeba_stretch_bend_dat
  use amoeba_torsion_torsion_mod, only : bcast_amoeba_tor_tor_dat

  implicit none

  call bcast_amoeba_bonds_dat
  call bcast_amoeba_ureyb_dat
  call bcast_amoeba_reg_angles_dat
  call bcast_amoeba_trig_angles_dat
  call bcast_amoeba_opbend_angles_dat
  call bcast_amoeba_torsions_dat
  call bcast_amoeba_stretch_torsions_dat
  call bcast_amoeba_pitorsions_dat
  call bcast_amoeba_stretch_bend_dat
  call bcast_amoeba_tor_tor_dat

  return

end subroutine bcast_amoeba_valence_dat
#endif

#ifdef MPI
!*******************************************************************************
!
! Subroutine:  bcast_amoeba_nonbond_dat
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine bcast_amoeba_nonbond_dat

  use pmemd_lib_mod
  use parallel_dat_mod
  use amoeba_multipoles_mod, only : bcast_amoeba_multipoles_dat
  use amoeba_adjust_mod, only : bcast_amoeba_adjust_dat
  use amoeba_vdw_mod, only : bcast_amoeba_vdw_dat
  use amoeba_induced_mod, only : bcast_amoeba_induced_dat
  use amoeba_direct_mod, only : bcast_amoeba_direct_dat
  use amoeba_recip_mod, only : bcast_amoeba_recip_dat
  use amoeba_self_mod, only : bcast_amoeba_self_dat

  implicit none

  call bcast_amoeba_multipoles_dat
  call bcast_amoeba_adjust_dat
  call bcast_amoeba_vdw_dat
  call bcast_amoeba_induced_dat

  ! The following routines are basically used to bcast a do_flag...

  call bcast_amoeba_direct_dat
  call bcast_amoeba_recip_dat
  call bcast_amoeba_self_dat

  return

end subroutine bcast_amoeba_nonbond_dat
#endif

!*******************************************************************************!
! Subroutine:  am_val_eval
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_val_eval(crd, frc, sander_vir, ebond, eangle, etors)

  use amoeba_bonds_mod, only : am_bonds_eval
  use amoeba_ureyb_mod, only : am_ureyb_eval
  use amoeba_reg_angles_mod, only : am_reg_angles_eval
  use amoeba_trig_angles_mod, only : am_trig_angles_eval
  use amoeba_opbend_angles_mod, only : am_opbend_angles_eval
  use amoeba_torsions_mod, only : am_torsions_eval
  use amoeba_stretch_torsions_mod, only : am_stretch_torsions_eval
  use amoeba_pitorsions_mod, only : am_pitorsions_eval
  use amoeba_stretch_bend_mod, only : am_stretch_bend_eval
  use amoeba_torsion_torsion_mod, only : am_tor_tor_eval
  use mdin_amoeba_dat_mod, only : do_amoeba_valence, amoeba_verbose
  use mdin_ctrl_dat_mod, only : iamoeba
  use file_io_dat_mod
  use timers_mod
  use parallel_dat_mod

  implicit none

! Formal arguments:

  double precision, intent(in)          :: crd(3, *)
  double precision, intent(in out)      :: frc(3, *)
  double precision, intent(in out)      :: sander_vir(3)
  double precision, intent(in out)      :: ebond
  double precision, intent(in out)      :: eangle
  double precision, intent(in out)      :: etors

! Local variables:

  double precision                      :: ene(10)
  double precision                      :: vir_tensor(3, 3)

  ene = 0.d0
  vir_tensor(:, :) = 0.d0

#ifdef MPI
  if (master) then      ! temporary conditional!
#endif
  if (iamoeba .eq. 1 .and. do_amoeba_valence .eq. 1) then
    call am_bonds_eval(crd, frc, ene(1), vir_tensor)
    call am_ureyb_eval(crd, frc, ene(2), vir_tensor)

    call update_time(bond_time)

    call am_reg_angles_eval(crd, frc, ene(3), vir_tensor)
    call am_trig_angles_eval(crd, frc, ene(4), vir_tensor)
    call am_opbend_angles_eval(crd, frc, ene(5), vir_tensor)

    call update_time(angle_time)

    call am_torsions_eval(crd, frc, ene(6), vir_tensor)
    call am_pitorsions_eval(crd, frc, ene(7), vir_tensor)
    call am_stretch_bend_eval(crd, frc, ene(8), vir_tensor)
    call am_tor_tor_eval(crd, frc, ene(9), vir_tensor)
    call am_stretch_torsions_eval(crd, frc, ene(10), vir_tensor)

    call update_time(dihedral_time)
  end if
#ifdef MPI
  end if                ! temporary conditional!
#endif

  if (amoeba_verbose .ne. 0) then
#ifdef MPI
     dbl_mpi_send_buf(1:10) = ene(1:10)
     dbl_mpi_send_buf(11:13) = vir_tensor(:,1)
     dbl_mpi_send_buf(14:16) = vir_tensor(:,2)
     dbl_mpi_send_buf(17:19) = vir_tensor(:,3)

     call mpi_reduce(dbl_mpi_send_buf, dbl_mpi_recv_buf, 19, &
                     mpi_double_precision, mpi_sum, 0, mpi_comm_world, &
                     err_code_mpi)

     if (master) then
       ene(1:10) = dbl_mpi_recv_buf(1:10)
       vir_tensor(:,1) = dbl_mpi_recv_buf(11:13)
       vir_tensor(:,2) = dbl_mpi_recv_buf(14:16)
       vir_tensor(:,3) = dbl_mpi_recv_buf(17:19)
#endif
       write(mdout, '(a, 3(1x, e16.8))') &
         'valence energies: bond,ureyb,angle ', ene(1), ene(2), ene(3)

       write(mdout, '(a, 3(1x, f14.4))') &
         'valence energies: trangle,opbend,tor ', ene(4), ene(5), ene(6)

       write(mdout, '(a, 3(1x, f14.4))') &
         'valence energies: pitors,strbend,tortor ', ene(7), ene(8), ene(9)

       write(mdout, '(a, 1x, f14.4)') &
         'valence energies: strtor ', ene(10)

       write(mdout, '(a, 3(1x, f14.4))') &
         'valence vir = ', vir_tensor(1, 1), vir_tensor(1, 2), vir_tensor(1, 3)

       write(mdout, '(a, 3(1x, f14.4))') &
         'valence vir = ', vir_tensor(2, 1), vir_tensor(2, 2), vir_tensor(2, 3)

       write(mdout, '(a, 3(1x, f14.4))') &
         'valence vir = ', vir_tensor(3, 1), vir_tensor(3, 2), vir_tensor(3, 3)

#ifdef MPI
      ! Restore original single process values in master.  This will
      ! be reduced again later.  This is amoeba_verbose-only code (debug).
       ene(1:10) = dbl_mpi_send_buf(1:10)
       vir_tensor(:,1) = dbl_mpi_send_buf(11:13)
       vir_tensor(:,2) = dbl_mpi_send_buf(14:16)
       vir_tensor(:,3) = dbl_mpi_send_buf(17:19)
    end if
#endif
  end if

  ebond = ene(1) ! bonds energy
  eangle = ene(2) + ene(3) + ene(4) + ene(5) + ene(8) ! angle energy
  etors = ene(6) + ene(7) + ene(9) + ene(10) ! torsion energy
  sander_vir(1) = sander_vir(1) + vir_tensor(1, 1)
  sander_vir(2) = sander_vir(2) + vir_tensor(2, 2)
  sander_vir(3) = sander_vir(3) + vir_tensor(3, 3)

  return

end subroutine am_val_eval

!*******************************************************************************!
! Subroutine:  am_nonbond_eval
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_nonbond_eval(atm_cnt, crd, frc, img_frc, sander_vir, evdw, eelt, &
                           epolar, evdw_14, diprms, dipiter, netfrcs)

  use mdin_amoeba_dat_mod, only : do_amoeba_nonbond, amoeba_verbose
  use mdin_ctrl_dat_mod, only : iamoeba

  use amoeba_multipoles_mod, only : am_mpole_local_to_global, &
                                    torque_field, &
                                    global_multipole, &
                                    am_mpole_torque_to_force

  use amoeba_induced_mod, only : am_induced_eval
  use amoeba_adjust_mod, only : num_adjust_list
  use amoeba_recip_mod, only : amoeba_recip_step_alloc_dat, &
                               amoeba_recip_step_dealloc_dat
  use file_io_dat_mod
  use parallel_dat_mod

  implicit none

! Formal arguments:

  integer, intent(in)                   :: atm_cnt
  double precision, intent(in)          :: crd(3, *)
  double precision, intent(in out)      :: frc(3, *)
  double precision, intent(in out)      :: img_frc(3, *)
  double precision, intent(in out)      :: sander_vir(3)
  double precision, intent(out)         :: evdw
  double precision, intent(out)         :: eelt
  double precision, intent(out)         :: epolar
  double precision, intent(out)         :: evdw_14
  double precision, intent(out)         :: dipiter
  double precision, intent(out)         :: diprms
  double precision, intent(out)         :: netfrcs(3)

! Local variables:

  integer                               :: j, k
  integer                               :: num_ints, num_reals  ! not yet used.
  double precision                      :: vir_tensor(3, 3)

  num_ints = 0
  num_reals = 0

  evdw = 0.d0
  eelt = 0.d0
  epolar = 0.d0
  evdw_14 = 0.d0
  vir_tensor(:,:) = 0.d0
  netfrcs(:) = 0.d0             ! In case the code below does not execute...

  if (iamoeba .eq. 1 .and. do_amoeba_nonbond .eq. 1) then

    torque_field(1:10,1:atm_cnt) = 0.d0
    global_multipole(1:10,1:atm_cnt) = 0.d0

    call am_mpole_local_to_global(crd)

    call amoeba_recip_step_alloc_dat(num_ints, num_reals)

    call am_induced_eval(atm_cnt, crd, diprms, dipiter, num_adjust_list)

    ! The frc array must be zero'd prior to entry into am_nonbond_ene_frc(),
    ! or the netfrcs calculation will be incorrect.
    call am_nonbond_ene_frc(atm_cnt, crd, eelt, epolar, evdw, &
                            evdw_14, frc, img_frc, vir_tensor, netfrcs)

    ! Add the torque contributions

    call am_mpole_torque_to_force(atm_cnt, crd, frc, vir_tensor)
    
    ! call  dump_dipoles(frc, atm_cnt, 40)
    ! if (atm_cnt > 0) stop

    if (amoeba_verbose .ne. 0) then
#ifdef MPI
      dbl_mpi_send_buf(1:3) = vir_tensor(:,1)
      dbl_mpi_send_buf(4:6) = vir_tensor(:,2)
      dbl_mpi_send_buf(7:9) = vir_tensor(:,3)

      call mpi_reduce(dbl_mpi_send_buf, dbl_mpi_recv_buf, 9, &
                      mpi_double_precision, mpi_sum, 0, mpi_comm_world, &
                      err_code_mpi)

      if (master) then
        vir_tensor(:,1) = dbl_mpi_recv_buf(1:3)
        vir_tensor(:,2) = dbl_mpi_recv_buf(4:6)
        vir_tensor(:,3) = dbl_mpi_recv_buf(7:9)
#endif
        write(mdout, '(a, 3(1x, g16.8))') &
          'nonbond vir = ', vir_tensor(1, 1), vir_tensor(1, 2), vir_tensor(1, 3)

        write(mdout, '(a, 3(1x, g16.8))') &
          'nonbond vir = ', vir_tensor(2, 1), vir_tensor(2, 2), vir_tensor(2, 3)

        write(mdout, '(a, 3(1x, g16.8))') &
          'nonbond vir = ', vir_tensor(3, 1), vir_tensor(3, 2), vir_tensor(3, 3)

#ifdef MPI
        ! Restore original single process values in master.  This will
        ! be reduced again later.  This is amoeba_verbose-only code (debug).
        vir_tensor(:,1) = dbl_mpi_send_buf(1:3)
        vir_tensor(:,2) = dbl_mpi_send_buf(4:6)
        vir_tensor(:,3) = dbl_mpi_send_buf(7:9)
      end if
#endif
    end if
    
    call amoeba_recip_step_dealloc_dat(num_ints, num_reals)

  end if

  sander_vir(1) = sander_vir(1) + vir_tensor(1, 1)
  sander_vir(2) = sander_vir(2) + vir_tensor(2, 2)
  sander_vir(3) = sander_vir(3) + vir_tensor(3, 3)

  return

end subroutine am_nonbond_eval

#endif /* AMOEBA */
end module amoeba_interface_mod
#ifdef AMOEBA
!*******************************************************************************!
! Subroutine:  am_nonbond_perm_fields
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_nonbond_perm_fields(atm_cnt, is_polarizable, crd, &
                                  dip_field_d, dip_field_p, adj_dip_dip_tensor)

  use amoeba_recip_mod, only : am_recip_perm_field
  use amoeba_direct_mod, only : am_direct_permfield
  use amoeba_adjust_mod, only : am_adjust_permfield
  use amoeba_self_mod, only : am_self_permfield
  use img_mod
  use nb_pairlist_mod

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt
  logical, intent(in)           :: is_polarizable(*)
  double precision, intent(in)  :: crd(3, *)
  double precision, intent(out) :: dip_field_d(3, atm_cnt)
  double precision, intent(out) :: dip_field_p(3, atm_cnt)
  double precision, intent(out) :: adj_dip_dip_tensor(6, *)

! Local variables:

  double precision              :: cart_dipole_field(3, atm_cnt)

  cart_dipole_field(:,:) = 0.d0

  call am_recip_perm_field(atm_cnt, crd, cart_dipole_field)

  call am_direct_permfield(cart_dipole_field, gbl_ipairs, gbl_img_crd, &
                           gbl_tranvec, gbl_img_atm_map)

  call am_adjust_permfield(crd, dip_field_d, dip_field_p, adj_dip_dip_tensor)

  call am_self_permfield(atm_cnt, dip_field_d, dip_field_p)
  
  call am_induced_add_cart_to_dip(atm_cnt, is_polarizable, cart_dipole_field, &
                                  dip_field_d, dip_field_p)

  return

end subroutine am_nonbond_perm_fields

!*******************************************************************************!
! Subroutine:  am_nonbond_dip_dip_fields
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_nonbond_dip_dip_fields(atm_cnt, crd, ind_dip_d, ind_dip_p, &
                                     adj_dip_dip_tensor, &
                                     dip_field_d, dip_field_p)

  use amoeba_recip_mod, only : am_recip_dipole_field
  use amoeba_direct_mod, only : am_direct_dip_dip_field
  use amoeba_adjust_mod, only : am_adjust_dip_dip_fields
  use amoeba_self_mod, only : am_self_dipole_field

  implicit none

! Formal arguments:

  integer, intent(in)           :: atm_cnt
  double precision, intent(in)  :: crd(3, *)
  double precision, intent(in)  :: ind_dip_d(3, *)
  double precision, intent(in)  :: ind_dip_p(3, *)
  double precision, intent(in)  :: adj_dip_dip_tensor(6, *)
  double precision, intent(out) :: dip_field_d(3, atm_cnt)
  double precision, intent(out) :: dip_field_p(3, atm_cnt)

  dip_field_d(:,:) = 0.d0
  dip_field_p(:,:) = 0.d0

  call am_recip_dipole_field(atm_cnt, crd, ind_dip_d, ind_dip_p, &
                             dip_field_d, dip_field_p)

  call am_direct_dip_dip_field(ind_dip_d, ind_dip_p, dip_field_d, dip_field_p)

  call am_adjust_dip_dip_fields(adj_dip_dip_tensor, ind_dip_d, ind_dip_p, &
                                dip_field_d, dip_field_p)

  call am_self_dipole_field(atm_cnt, ind_dip_d, ind_dip_p, dip_field_d, &
                            dip_field_p)

  return

end subroutine am_nonbond_dip_dip_fields

!*******************************************************************************!
! Subroutine:  am_nonbond_ene_frc
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_nonbond_ene_frc(atm_cnt, crd, ene_perm, ene_ind, ene_vdw, &
                              ene_vdw_14, frc, img_frc, vir_tensor, netfrcs)

  use amoeba_recip_mod, only : am_recip_ene_frc
  use amoeba_direct_mod, only : am_direct_ene_frc
  use amoeba_adjust_mod, only : am_adjust_ene_frc
  use amoeba_self_mod, only : am_self_ene_torque
  use amoeba_vdw_mod, only : am_vdw_longrange_ene
  use amoeba_induced_mod, only : ind_dip_d, ind_dip_p
  use mdin_amoeba_dat_mod, only : amoeba_verbose
  use file_io_dat_mod
  use img_mod
  use nb_pairlist_mod
  use parallel_dat_mod

  implicit none

! Formal arguments:

  integer, intent(in)                   :: atm_cnt
  double precision, intent(in)          :: crd(3, *)
  double precision, intent(out)         :: ene_perm
  double precision, intent(out)         :: ene_ind
  double precision, intent(out)         :: ene_vdw
  double precision, intent(out)         :: ene_vdw_14
  double precision, intent(in out)      :: frc(3, *)
  double precision, intent(in out)      :: img_frc(3, *)
  double precision, intent(in out)      :: vir_tensor(3, 3)
  double precision, intent(out)         :: netfrcs(3)

! Local variables:

  double precision                      :: e_rec_perm
  double precision                      :: e_rec_ind
  double precision                      :: e_dir_perm
  double precision                      :: e_dir_ind
  double precision                      :: e_adj_perm
  double precision                      :: e_adj_ind
  double precision                      :: e_self_perm
  double precision                      :: e_self_ind
  double precision                      :: e_dir_vdw
  double precision                      :: e_adj_vdw
  double precision                      :: e_rec_vdw

  ! NOTE: There is an assumption here that on entry the img_frc array is zero'd.
  !       If it is not, the net force calculation will be incorrect!

  call am_recip_ene_frc(atm_cnt, crd, ind_dip_d, ind_dip_p, e_rec_perm, &
                        e_rec_ind, img_frc, vir_tensor, netfrcs)

  call am_direct_ene_frc(gbl_ipairs, gbl_img_crd, gbl_tranvec, crd, &
                         ind_dip_d, ind_dip_p, e_dir_perm, e_dir_ind, &
                         e_dir_vdw, frc, img_frc, vir_tensor, gbl_img_atm_map)

  call am_adjust_ene_frc(crd, ind_dip_d, ind_dip_p, &
                         e_adj_perm, e_adj_ind, e_adj_vdw, frc, vir_tensor)

  call am_self_ene_torque(atm_cnt, ind_dip_d, ind_dip_p, &
                          e_self_perm, e_self_ind)

#ifdef MPI
  if (master) then
    call am_vdw_longrange_ene(e_rec_vdw, vir_tensor)
  else
    e_rec_vdw = 0.d0
  end if
#else
  call am_vdw_longrange_ene(e_rec_vdw, vir_tensor)
#endif

  if (amoeba_verbose .ne. 0) then
#ifdef MPI
    dbl_mpi_send_buf(1) =  e_rec_perm
    dbl_mpi_send_buf(2) =  e_dir_perm
    dbl_mpi_send_buf(3) =  e_adj_perm
    dbl_mpi_send_buf(4) =  e_self_perm
    dbl_mpi_send_buf(5) =  e_rec_ind
    dbl_mpi_send_buf(6) =  e_dir_ind
    dbl_mpi_send_buf(7) =  e_adj_ind
    dbl_mpi_send_buf(8) =  e_self_ind
    dbl_mpi_send_buf(9) =  e_dir_vdw
    dbl_mpi_send_buf(10) = e_adj_vdw
    dbl_mpi_send_buf(11) = e_rec_vdw

    call mpi_reduce(dbl_mpi_send_buf, dbl_mpi_recv_buf, 11, &
                    mpi_double_precision, mpi_sum, 0, mpi_comm_world, &
                    err_code_mpi)

    if (master) then
      e_rec_perm = dbl_mpi_recv_buf(1)
      e_dir_perm = dbl_mpi_recv_buf(2)
      e_adj_perm = dbl_mpi_recv_buf(3)
      e_self_perm = dbl_mpi_recv_buf(4)
      e_rec_ind = dbl_mpi_recv_buf(5)
      e_dir_ind = dbl_mpi_recv_buf(6)
      e_adj_ind = dbl_mpi_recv_buf(7)
      e_self_ind = dbl_mpi_recv_buf(8)
      e_dir_vdw = dbl_mpi_recv_buf(9)
      e_adj_vdw = dbl_mpi_recv_buf(10)
      e_rec_vdw = dbl_mpi_recv_buf(11)
#endif
      write(mdout, '(a, /, 4(1x, f14.4))') &
        'e_rec_perm,e_dir_perm,e_adj_perm,e_self_perm = ', &
        e_rec_perm, e_dir_perm, e_adj_perm, e_self_perm

      write(mdout, '(a, /, 4(1x, f14.4))') &
        'e_rec_ind,e_dir_ind,e_adj_ind,e_self_ind = ', &
        e_rec_ind, e_dir_ind, e_adj_ind, e_self_ind

      write(mdout, '(a, /, 3(1x, f14.4))') &
        'e_dir_vdw,e_adj_vdw,e_rec_vdw = ', &
        e_dir_vdw, e_adj_vdw, e_rec_vdw

#ifdef MPI
      ! Restore original single process values in master.  This will
      ! be reduced again later.  This is amoeba_verbose-only code (debug).
      e_rec_perm = dbl_mpi_send_buf(1)
      e_dir_perm = dbl_mpi_send_buf(2)
      e_adj_perm = dbl_mpi_send_buf(3)
      e_self_perm = dbl_mpi_send_buf(4)
      e_rec_ind = dbl_mpi_send_buf(5)
      e_dir_ind = dbl_mpi_send_buf(6)
      e_adj_ind = dbl_mpi_send_buf(7)
      e_self_ind = dbl_mpi_send_buf(8)
      e_dir_vdw = dbl_mpi_send_buf(9)
      e_adj_vdw = dbl_mpi_send_buf(10)
      e_rec_vdw = dbl_mpi_send_buf(11)
    end if
#endif
  end if

  ene_perm = e_rec_perm + e_dir_perm + e_adj_perm + e_self_perm
  ene_ind = e_rec_ind + e_dir_ind + e_adj_ind + e_self_ind
  ene_vdw = e_dir_vdw + e_rec_vdw
  ene_vdw_14 = e_adj_vdw

  return

end subroutine am_nonbond_ene_frc

!*******************************************************************************!
! Subroutine:  am_nonbond_set_user_bit
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_nonbond_set_user_bit(do_recip, do_adjust, do_direct, do_self, &
                                   do_vdw, do_induce)

  use amoeba_recip_mod, only : am_recip_zero_flag, am_recip_set_user_bit
  use amoeba_adjust_mod, only : am_adjust_zero_flag, am_adjust_set_user_bit
  use amoeba_direct_mod, only : am_direct_zero_flag, am_direct_set_user_bit
  use amoeba_self_mod, only : am_self_zero_flag, am_self_set_user_bit
  use amoeba_vdw_mod, only : am_vdw_zero_flag, am_vdw_set_user_bit
  use amoeba_induced_mod, only : am_induced_zero_flag, am_induced_set_user_bit

  implicit none

! Formal arguments:

  integer, intent(in)  :: do_recip
  integer, intent(in)  :: do_adjust
  integer, intent(in)  :: do_direct
  integer, intent(in)  :: do_self
  integer, intent(in)  :: do_vdw
  integer, intent(in)  :: do_induce
                          
  call am_recip_zero_flag
  call am_adjust_zero_flag
  call am_direct_zero_flag
  call am_self_zero_flag
  call am_vdw_zero_flag
  call am_induced_zero_flag

  call am_recip_set_user_bit(do_recip)
  call am_adjust_set_user_bit(do_adjust)
  call am_direct_set_user_bit(do_direct)
  call am_self_set_user_bit(do_self)
  call am_vdw_set_user_bit(do_vdw)
  call am_induced_set_user_bit(do_induce)

  return

end subroutine am_nonbond_set_user_bit

!*******************************************************************************!
! Subroutine:  am_val_set_user_bit
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_val_set_user_bit(do_bond, do_ureyb, do_reg_angle, &
                               do_trig_angle, do_opbend_angle, do_torsions, &
                               do_str_torsions, do_pitorsions,  &
                               do_stretch_bend, do_torsion_torsion)

  use amoeba_bonds_mod, only : am_bonds_zero_flag, &
                               am_bonds_set_user_bit
  use amoeba_ureyb_mod, only : am_ureyb_zero_flag, &
                               am_ureyb_set_user_bit
  use amoeba_reg_angles_mod, only : am_reg_angles_zero_flag, &
                                    am_reg_angles_set_user_bit
  use amoeba_trig_angles_mod, only : am_trig_angles_zero_flag, &
                                     am_trig_angles_set_user_bit
  use amoeba_opbend_angles_mod, only : am_opbend_angles_zero_flag, &
                                       am_opbend_angles_set_user_bit
  use amoeba_torsions_mod, only : am_torsions_zero_flag, &
                                  am_torsions_set_user_bit
  use amoeba_stretch_torsions_mod, only : am_stretch_torsions_zero_flag, &
                                          am_stretch_torsions_suser_bit
  use amoeba_pitorsions_mod, only : am_pitorsions_zero_flag, &
                                    am_pitorsions_set_user_bit
  use amoeba_stretch_bend_mod, only : am_stretch_bend_zero_flag, &
                                      am_stretch_bend_set_user_bit
  use amoeba_torsion_torsion_mod, only : am_tor_tor_zero_flag, &
                                         am_tor_tor_set_user_bit

  implicit none

! Formal arguments:

  integer, intent(in)   :: do_bond
  integer, intent(in)   :: do_ureyb
  integer, intent(in)   :: do_reg_angle
  integer, intent(in)   :: do_trig_angle
  integer, intent(in)   :: do_opbend_angle
  integer, intent(in)   :: do_torsions
  integer, intent(in)   :: do_pitorsions
  integer, intent(in)   :: do_str_torsions
  integer, intent(in)   :: do_stretch_bend
  integer, intent(in)   :: do_torsion_torsion

  call am_bonds_zero_flag
  call am_ureyb_zero_flag
  call am_reg_angles_zero_flag
  call am_trig_angles_zero_flag
  call am_opbend_angles_zero_flag
  call am_torsions_zero_flag
  call am_stretch_torsions_zero_flag
  call am_pitorsions_zero_flag
  call am_stretch_bend_zero_flag
  call am_tor_tor_zero_flag

  call am_bonds_set_user_bit(do_bond)
  call am_ureyb_set_user_bit(do_ureyb)
  call am_reg_angles_set_user_bit(do_reg_angle)
  call am_trig_angles_set_user_bit(do_trig_angle)
  call am_opbend_angles_set_user_bit(do_opbend_angle)
  call am_torsions_set_user_bit(do_torsions)
  call am_stretch_torsions_suser_bit(do_str_torsions)
  call am_pitorsions_set_user_bit(do_pitorsions)
  call am_stretch_bend_set_user_bit(do_stretch_bend)
  call am_tor_tor_set_user_bit(do_torsion_torsion)

  return

end subroutine am_val_set_user_bit

!*******************************************************************************!
! Subroutine:  amoeba_get_numlist
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_get_numlist(header, nf, num_list)

  use file_io_dat_mod
  use nextprmtop_section_mod

  implicit none

! Formal arguments:

  character(len = *), intent(in)        :: header
  integer, intent(in)                   :: nf
  integer, intent(out)                  :: num_list

! Local variables:

  integer                               :: ier, iok, n, ionerr
  character(len = 80)                   :: fmt
  character(len = 80)                   :: fmtin, dtype

  fmtin = '(10I8)'
  dtype = header // 'NUM_LIST'
  ionerr = 1            ! not fatal if missing

  call nxtsec(nf, mdout, ionerr, fmtin, dtype, fmt, iok)

  if (iok .eq. 0) then          ! this data type found in prmtop
    read(nf, fmt) num_list
  else                          ! either old style prmtop or data not found
    num_list = 0                ! upon return this will invalidate valence_term
  end if

  return

end subroutine amoeba_get_numlist

!*******************************************************************************!
! Subroutine:  amoeba_read_list_data
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_read_list_data(header, nf, dim1, num_list, list)

  use file_io_dat_mod
  use nextprmtop_section_mod

  implicit none

! Formal arguments:

  character(len = *), intent(in)        :: header
  integer, intent(in)                   :: nf, dim1, num_list
  integer, intent(out)                  :: list(dim1, num_list)

! Local variables:

  integer                               :: iok, ionerr, j, k
  character(len = 80)                   :: fmt
  character(len = 80)                   :: fmtin, dtype

  ionerr = 0                    ! fatal if missing
  fmtin = '(10I8)'
  dtype = header // 'LIST'
  call nxtsec(nf, mdout, ionerr, fmtin, dtype, fmt, iok)
  read(nf, fmt) ((list(j, k), j = 1, dim1), k = 1, num_list)

  return

end subroutine amoeba_read_list_data

!*******************************************************************************!
! Subroutine:  amoeba_read_real_list_data
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_read_real_list_data(header, nf, dim1, num_list, list)

  use file_io_dat_mod
  use nextprmtop_section_mod

  implicit none

! Formal arguments:

  character(len = *), intent(in)        :: header
  integer, intent(in)                   :: nf, dim1, num_list
  double precision, intent(out)         :: list(dim1, num_list)

! Local variables:

  integer                               :: iok, ionerr, j, k
  character(len = 80)                   :: fmt
  character(len = 80)                   :: fmtin, dtype

  ionerr = 0                    ! fatal if missing
  fmtin = '(5E16.8)'
  dtype = header // 'LIST'

  call nxtsec(nf, mdout, ionerr, fmtin, dtype, fmt, iok)
  read(nf, fmt) ((list(j, k), j = 1, dim1), k = 1, num_list)

  return

end subroutine amoeba_read_real_list_data

!*******************************************************************************!
! Subroutine:  amoeba_read_real_scalar
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_read_real_scalar(flag, nf, scalar_value)

  use file_io_dat_mod
  use nextprmtop_section_mod

  implicit none

! Formal arguments:

  character(len = *), intent(in)        :: flag
  integer, intent(in)                   :: nf
  double precision, intent(out)         :: scalar_value

! Local variables:

  integer                               :: iok, ionerr, j, k
  character(len = 80)                   :: fmt
  character(len = 80)                   :: fmtin, dtype

  ionerr = 0            ! fatal if missing
  fmtin = '(E16.8)'
  call nxtsec(nf, mdout, ionerr, fmtin, flag, fmt, iok)
  read(nf, fmt) scalar_value

  return

end subroutine amoeba_read_real_scalar

!*******************************************************************************!
! Subroutine:  amoeba_get_startlist_endlist
!
! Description: <TBS>
!
!*******************************************************************************

subroutine amoeba_get_startlist_endlist(num_list, startlist, endlist, siztask)

  implicit none

! Formal arguments:

  integer, intent(in)   :: num_list
  integer, intent(out)  :: startlist, endlist, siztask

! Local variables:

  integer               :: piece
  integer               :: numtasks, mytaskid

  numtasks = 1  
  mytaskid = 0

  if (numtasks .gt. 1) then
    piece = num_list / numtasks
    startlist = mytaskid * piece + 1
    endlist   = mytaskid * piece + piece
    if (mytaskid .eq. (numtasks - 1)) endlist = num_list
    siztask = endlist-startlist + 1
  else
    startlist = 1
    endlist = num_list
    siztask = endlist-startlist + 1
  end if

  return

end subroutine amoeba_get_startlist_endlist

!*******************************************************************************!
! Subroutine:  array_add
!
! Description: <TBS>
!
!*******************************************************************************

subroutine array_add(a, b, num)

  implicit none

! Formal arguments:

  double precision, intent(in out)      :: a(*)
  double precision, intent(in)          :: b(*)
  integer, intent(in)                   :: num

! Local variables:

  integer                               :: n

  do n = 1, num
    a(n) = a(n) + b(n)
  end do

  return

end subroutine array_add

!*******************************************************************************!
! Subroutine:  array_copy
!
! Description: <TBS>
!
!*******************************************************************************

subroutine array_copy(a, b, num)

  implicit none

! Formal arguments:

  double precision, intent(in out)      :: a(1:num)
  double precision, intent(in)          :: b(1:num)
  integer, intent(in)                   :: num

  a(:) = b(:)

  return

end subroutine array_copy

!*******************************************************************************!
! Subroutine:  dump_3d_array
!
! Description: <TBS>
!
!*******************************************************************************

subroutine dump_3d_array(array, cnt, nf)

  implicit none

! Formal arguments:

  double precision      :: array(3, *)
  integer               :: cnt, nf

! Local variables:

  integer               :: j, n

  do n = 1, cnt
    write(nf, '(3f20.12)') (array(j, n), j = 1, 3)
  end do

  return

end subroutine dump_3d_array

#ifdef MPI
!*******************************************************************************!
! Subroutine:  reduce_dump_3d_array
!
! Description: <TBS>
!
!*******************************************************************************

subroutine reduce_dump_3d_array(array, cnt, nf)

  use parallel_dat_mod

  implicit none

! Formal arguments:

  double precision      :: array(3, *)
  integer               :: cnt, nf

! Local variables:

  integer               :: j, n
  double precision      :: outbuf(3, cnt)

  call mpi_allreduce(array, dbl_mpi_recv_buf, 3 * cnt, &
                     mpi_double_precision, mpi_sum, mpi_comm_world, &
                     err_code_mpi)
  call array_copy(outbuf, dbl_mpi_recv_buf, 3 * cnt)

  if (master) then
    do n = 1, cnt
      write(nf, '(3f20.12)') (outbuf(j, n), j = 1, 3)
    end do
  end if

  return

end subroutine reduce_dump_3d_array
#endif /* MPI */
#endif /* AMOEBA */
