#include "copyright.i"

!*******************************************************************************!
! Module: amoeba_torsion_torsion_mod
!
! Description: <TBS>
!
!*******************************************************************************

module amoeba_torsion_torsion_mod
#ifdef AMOEBA

  implicit none

  private

! Data that should be broadcast to slave processes from the master:

  integer, parameter    :: amoeba_tor_tor_int_cnt = 3

  integer                       do_amoeba_tor_tor_flag, num_list, num_params

  common / amoeba_tor_tor_int / do_amoeba_tor_tor_flag, num_list, num_params

  save  :: / amoeba_tor_tor_int /

  integer, allocatable, save            :: list(:,:)

  type angle_angle_functable
    integer                     :: dim1 = 0
    integer                     :: dim2 = 0
    double precision, pointer   :: angle1(:) => null()
    double precision, pointer   :: angle2(:) => null()
    double precision, pointer   :: func(:,:) => null()
    double precision, pointer   :: dfunc_dangle1(:,:) => null()
    double precision, pointer   :: dfunc_dangle2(:,:) => null()
    double precision, pointer   :: d2func_dangle1_dangle2(:,:) => null()
  end type  angle_angle_functable

  type(angle_angle_functable), allocatable, save  :: torsion_torsion_tbl(:) 

  double precision, parameter   :: pi = 3.14159265358979323846d0
  double precision, parameter   :: radians_to_degrees = 180.d0 / pi
  double precision, parameter   :: degrees_to_radians = pi / 180.d0

  ! BUGBUG - move to local storage:

  double precision, save                :: energy
  double precision, save                :: virial(3,3)

  public        am_tor_tor_zero_flag
  public        am_tor_tor_set_user_bit
  public        init_amoeba_tor_tor_dat
  public        am_tor_tor_eval
#ifdef MPI
  public        bcast_amoeba_tor_tor_dat
#endif

contains

!*******************************************************************************!
! Subroutine:  am_tor_tor_zero_flag
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_tor_tor_zero_flag
  implicit none
  do_amoeba_tor_tor_flag = 0
  return
end subroutine am_tor_tor_zero_flag

!*******************************************************************************!
! Function:  init_amoeba_tor_tor_dat
!
! Description: <TBS>
!
!*******************************************************************************

function init_amoeba_tor_tor_dat(num_ints, num_reals)

  use amoeba_flags_mod
  use file_io_dat_mod
  use parallel_dat_mod
  use pmemd_lib_mod

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

   integer, intent(in out)      :: num_ints, num_reals

! Local variables:

  integer               :: init_amoeba_tor_tor_dat
  integer               :: alloc_failed
  integer, allocatable  :: param_dims_lst(:,:)

  init_amoeba_tor_tor_dat = 0

  call amoeba_get_numlist('AMOEBA_TORSION_TORSION_', prmtop, num_list)

  if (num_list .le. 0) then
    do_amoeba_tor_tor_flag = ibclr(do_amoeba_tor_tor_flag, valid_bit)
    return
  end if

  call am_val_get_num_params('AMOEBA_TORSION_TORSION_', prmtop, num_params)

  if (num_params .le. 0) then
    do_amoeba_tor_tor_flag = ibclr(do_amoeba_tor_tor_flag, valid_bit)
    return
  end if

  allocate(param_dims_lst(2, num_params), stat = alloc_failed)
  if (alloc_failed .ne. 0) call setup_alloc_error

  call am_tor_tor_read_tortor_param_dims(param_dims_lst)

  call alloc_amoeba_tor_tor_mem(param_dims_lst, num_ints, num_reals)

  deallocate(param_dims_lst)

  call amoeba_read_list_data('AMOEBA_TORSION_TORSION_', prmtop, 6, &
                             num_list, list)

  call am_tor_tor_read_tortor_ftbl

  do_amoeba_tor_tor_flag = ibset(do_amoeba_tor_tor_flag, valid_bit)

  init_amoeba_tor_tor_dat = 1

  return

end function init_amoeba_tor_tor_dat

#ifdef MPI
!*******************************************************************************
!
! Subroutine:  bcast_amoeba_tor_tor_dat
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine bcast_amoeba_tor_tor_dat

  use pmemd_lib_mod
  use parallel_dat_mod
  use amoeba_flags_mod

  implicit none

  integer               :: alloc_failed
  integer               :: dim1, dim2, dimprod
  integer               :: n
  integer               :: num_ints, num_reals  ! returned values discarded
  integer, allocatable  :: param_dims_lst(:,:)

  call mpi_bcast(do_amoeba_tor_tor_flag, amoeba_tor_tor_int_cnt, &
                 mpi_integer, 0, mpi_comm_world, err_code_mpi)

  if (do_amoeba_tor_tor_flag .ne. proceed) return

  allocate(param_dims_lst(2, num_params), stat = alloc_failed)
  if (alloc_failed .ne. 0) call setup_alloc_error

  if (master) then
    do n = 1, num_params
      param_dims_lst(1, n) = torsion_torsion_tbl(n)%dim1
      param_dims_lst(2, n) = torsion_torsion_tbl(n)%dim2
    end do
  end if

  call mpi_bcast(param_dims_lst, 2 * num_params, mpi_integer, 0, &
                 mpi_comm_world, err_code_mpi)

  if (.not. master) then
    num_ints = 0
    num_reals = 0
    call alloc_amoeba_tor_tor_mem(param_dims_lst, num_ints, num_reals)
  end if

  if (num_list .gt. 0) then
    call mpi_bcast(list, 6 * num_list, mpi_integer, 0, &
                   mpi_comm_world, err_code_mpi)
  end if 

  if (num_params .gt. 0) then
    do n = 1, num_params
      ! The torsion_torsion_tbl(n)%dim1,2 elements were filled out on alloc.
      ! They are actually redundant because size() info is available...
      dim1 = param_dims_lst(1, n)
      dim2 = param_dims_lst(2, n)
      dimprod = dim1 * dim2
      call mpi_bcast(torsion_torsion_tbl(n)%angle1, dim1, &
                     mpi_double_precision, 0, mpi_comm_world, err_code_mpi)
      call mpi_bcast(torsion_torsion_tbl(n)%angle2, dim2, &
                     mpi_double_precision, 0, mpi_comm_world, err_code_mpi)
      call mpi_bcast(torsion_torsion_tbl(n)%func, dimprod, &
                     mpi_double_precision, 0, mpi_comm_world, err_code_mpi)
      call mpi_bcast(torsion_torsion_tbl(n)%dfunc_dangle1, dimprod, &
                     mpi_double_precision, 0, mpi_comm_world, err_code_mpi)
      call mpi_bcast(torsion_torsion_tbl(n)%dfunc_dangle2, dimprod, &
                     mpi_double_precision, 0, mpi_comm_world, err_code_mpi)
      call mpi_bcast(torsion_torsion_tbl(n)%d2func_dangle1_dangle2, dimprod, &
                     mpi_double_precision, 0, mpi_comm_world, err_code_mpi)
    end do
  end if 

  deallocate(param_dims_lst)

  return

end subroutine bcast_amoeba_tor_tor_dat
#endif /* MPI */

!*******************************************************************************
!
! Subroutine:  alloc_amoeba_tor_tor_mem
!
! Description: <TBS>
!              
!*******************************************************************************

subroutine alloc_amoeba_tor_tor_mem(param_dims_lst, num_ints, num_reals)

  use pmemd_lib_mod
  use parallel_dat_mod
  use amoeba_flags_mod

  implicit none

! Formal arguments:

  ! num_ints and num_reals are used to return allocation counts. Don't zero.

  integer                       :: param_dims_lst(2, num_params)
  integer, intent(in out)       :: num_ints, num_reals     

! Local variables:

  integer                       :: alloc_failed
  integer                       :: dim1, dim2
  integer                       :: n

  ! List has 5 atoms plus param ptr...

  allocate(list(6, num_list), stat = alloc_failed)
  if (alloc_failed .ne. 0) call setup_alloc_error
  num_ints = num_ints + size(list)

  allocate(torsion_torsion_tbl(num_params), stat = alloc_failed)
  if (alloc_failed .ne. 0) call setup_alloc_error
  num_ints = num_ints + num_params * 8 ! count ptrs as ints... 

  ! With the num_params and the param_dims_lst, we have enough info to
  ! allocate all the table members.  Also fill in the dimensions.

  do n = 1, num_params
    dim1 = param_dims_lst(1, n)
    dim2 = param_dims_lst(2, n)
    torsion_torsion_tbl(n)%dim1 = dim1
    torsion_torsion_tbl(n)%dim2 = dim2
    allocate(torsion_torsion_tbl(n)%angle1(dim1), &
             torsion_torsion_tbl(n)%angle2(dim2), &
             torsion_torsion_tbl(n)%func(dim1, dim2), &
             torsion_torsion_tbl(n)%dfunc_dangle1(dim1, dim2), &
             torsion_torsion_tbl(n)%dfunc_dangle2(dim1, dim2), &
             torsion_torsion_tbl(n)%d2func_dangle1_dangle2(dim1, dim2), &
             stat = alloc_failed)
    if (alloc_failed .ne. 0) call setup_alloc_error
    num_reals = num_reals + dim1 + dim2 + 4 * (dim1 * dim2)
  end do

  return

end subroutine alloc_amoeba_tor_tor_mem

!*******************************************************************************!
! Subroutine:  am_tor_tor_read_tortor_param_dims
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_tor_tor_read_tortor_param_dims(param_dims_lst)

  use nextprmtop_section_mod
  use file_io_dat_mod
  use parallel_dat_mod

  implicit none

! Formal arguments:

  integer, intent(out)  :: param_dims_lst(2, num_params)

! Local variables:

  integer               :: iok, ionerr
  integer               :: n
  character(len = 2)    :: word
  character(len = 80)   :: fmt
  character(len = 80)   :: fmtin, dtype

  ionerr = 0 ! fatal if missing
  fmtin = '(10I8)'

  do n = 1, num_params
    write(word, '(i2.2)') n
    dtype = 'AMOEBA_TORSION_TORSION_TORTOR_TABLE_' // word // '_DIMS'
    call nxtsec(prmtop,  mdout,  ionerr, fmtin,  dtype,  fmt,  iok)
    read(prmtop, fmt) param_dims_lst(1, n), param_dims_lst(2, n)
  end do

  return

end subroutine am_tor_tor_read_tortor_param_dims

!*******************************************************************************!
! Subroutine:  am_tor_tor_read_tortor_ftbl
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_tor_tor_read_tortor_ftbl

  use nextprmtop_section_mod
  use file_io_dat_mod
  use parallel_dat_mod

  implicit none

! Local variables:

  integer                               :: iok, ionerr
  integer                               :: n
  character(len = 2)                    :: word
  character(len = 80)                   :: fmt
  character(len = 80)                   :: fmtin, dtype

  ionerr = 0 ! fatal if missing
  fmtin = '(5E16.8)'

  ! When the arrays were allocated, dimensions were associated with them.
  ! This is sort of slick...

  do n = 1, num_params

    write(word, '(i2.2)') n
    
    dtype = 'AMOEBA_TORSION_TORSION_TORTOR_TABLE_' // word // '_' // &
            'ANGLE1'
    call nxtsec(prmtop, mdout, ionerr, fmtin, dtype, fmt, iok)
    read(prmtop, fmt) torsion_torsion_tbl(n)%angle1

    dtype = 'AMOEBA_TORSION_TORSION_TORTOR_TABLE_' // word // '_' // &
            'ANGLE2'
    call nxtsec(prmtop, mdout, ionerr, fmtin, dtype, fmt, iok)
    read(prmtop, fmt) torsion_torsion_tbl(n)%angle2

    dtype = 'AMOEBA_TORSION_TORSION_TORTOR_TABLE_' // word // '_' // &
            'FUNC'
    call nxtsec(prmtop, mdout, ionerr, fmtin, dtype, fmt, iok)
    read(prmtop, fmt) torsion_torsion_tbl(n)%func

    dtype = 'AMOEBA_TORSION_TORSION_TORTOR_TABLE_' // word // '_' // &
            'DFUNC_DANGLE1'
    call nxtsec(prmtop, mdout, ionerr, fmtin, dtype, fmt, iok)
    read(prmtop, fmt) torsion_torsion_tbl(n)%dfunc_dangle1

    dtype = 'AMOEBA_TORSION_TORSION_TORTOR_TABLE_' // word // '_' // &
            'DFUNC_DANGLE2'
    call nxtsec(prmtop, mdout, ionerr, fmtin, dtype, fmt, iok)
    read(prmtop, fmt) torsion_torsion_tbl(n)%dfunc_dangle2

    dtype = 'AMOEBA_TORSION_TORSION_TORTOR_TABLE_' // word // '_' &
            // 'D2FUNC_DANGLE1_DANGLE2'
    call nxtsec(prmtop, mdout, ionerr, fmtin, dtype, fmt, iok)
    read(prmtop, fmt) torsion_torsion_tbl(n)%d2func_dangle1_dangle2

  end do

  return

end subroutine am_tor_tor_read_tortor_ftbl

!*******************************************************************************!
! Subroutine:  am_tor_tor_set_user_bit
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_tor_tor_set_user_bit(do_this)

  use amoeba_flags_mod

  implicit none

! Formal arguments:

  integer, intent(in)   :: do_this

  call set_user_bit(do_this, do_amoeba_tor_tor_flag)

  return

end subroutine am_tor_tor_set_user_bit

!*******************************************************************************!
! Subroutine:  am_tor_tor_eval
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_tor_tor_eval(crd, frc, ene, vir)

  use amoeba_flags_mod

  implicit none

! Formal arguments:

  double precision, intent(in)          :: crd(3, *)
  double precision, intent(in out)      :: frc(3, *)
  double precision, intent(out)         :: ene
  double precision, intent(in out)      :: vir(3, 3)

! Local variables:

  double precision                      :: arg1(num_list)
  double precision                      :: arg2(num_list)
  double precision                      :: darg1_dcrd(12, num_list)
  double precision                      :: darg2_dcrd(12, num_list)
  double precision                      :: func(num_list)
  double precision                      :: dfunc_darg1(num_list)
  double precision                      :: dfunc_darg2(num_list)

  energy = 0.d0
  virial(:, :) = 0.d0

  if (do_amoeba_tor_tor_flag .ne. proceed) return

  call am_tor_tor_get_args(list, crd, arg1, arg2, darg1_dcrd, darg2_dcrd)

  call am_tor_tor_func(list, arg1, arg2, torsion_torsion_tbl, &
                       func, dfunc_darg1, dfunc_darg2)

  call am_tor_tor_get_ene_frc(list, crd, func, dfunc_darg1, dfunc_darg2, &
                              darg1_dcrd, darg2_dcrd, frc)
  ene = energy

  vir(:, :) = vir(:, :) + virial(:, :)

  return

end subroutine am_tor_tor_eval

!*******************************************************************************!
! Subroutine:  am_tor_tor_get_args
!
! Description:
!
! This routine calculates torsion-torsion function arguments and their
!  derivatives with  respect to atomic positions of atom i, j, k, l, m.
!
! INPUT variables:
!    crd the atomic coord array
!    list: 6 x ntortor array giving for each torsion-torsion 
!           the index of the first, second, third, fourth and fifth atoms
!           and the param table pointer
! OUTPUT variables:
!    for each torsion-torsion in list
!    arg1--the torsion angle of i, j, k, l
!    arg2--the torsion angle of j, k, l, m
!    darg1_dcrdijkl   derivs of arg1 wrt crds of atom i, j, k, l
!    darg2_dcrdjklm   derivs of arg2 wrt crds of atom j, k, l, m
!
!*******************************************************************************

subroutine am_tor_tor_get_args(list, crd, arg1, arg2,   &
                               darg1_dcrdijkl, darg2_dcrdjklm)

  implicit none

! Formal arguments:

  integer, intent(in)           :: list(6, *) !5 atoms plus param ptr
  double precision, intent(in)  :: crd(3, *)
  double precision, intent(out) :: arg1(*)
  double precision, intent(out) :: arg2(*)
  double precision, intent(out) :: darg1_dcrdijkl(12, *)
  double precision, intent(out) :: darg2_dcrdjklm(12, *)

! Local variables:

  integer                       :: i, j, k, l, m, n, p
  double precision              :: crd_abcd(12), gradphi_abcd(12)
  double precision              :: cosphi, sinphi
  double precision              :: phi

  do n = 1, num_list

    i = list(1, n)
    j = list(2, n)
    k = list(3, n)
    l = list(4, n)
    m = list(5, n)

    do p = 1, 3
      crd_abcd(p) = crd(p, i)
      crd_abcd(p + 3) = crd(p, j)
      crd_abcd(p + 6) = crd(p, k)
      crd_abcd(p + 9) = crd(p, l)
    end do

    call am_val_geom_torsion(crd_abcd, gradphi_abcd, cosphi, sinphi)

    phi = radians_to_degrees * acos(cosphi)

    if (sinphi .ge. 0.d0) then
      arg1(n) = phi
    else
      arg1(n) = -phi
    end if

    do p = 1, 12
      darg1_dcrdijkl(p, n) = radians_to_degrees * gradphi_abcd(p)
    end do

    do p = 1, 3
      crd_abcd(p) =   crd(p, j)
      crd_abcd(p + 3) = crd(p, k)
      crd_abcd(p + 6) = crd(p, l)
      crd_abcd(p + 9) = crd(p, m)
    end do

    call am_val_geom_torsion(crd_abcd, gradphi_abcd, cosphi, sinphi)

    phi = radians_to_degrees * acos(cosphi)

    if (sinphi .ge. 0.d0) then
      arg2(n) = phi
    else
      arg2(n) = -phi
    end if

    do p = 1, 12
      darg2_dcrdjklm(p, n) = radians_to_degrees * gradphi_abcd(p)
    end do

  end do

  return

end subroutine am_tor_tor_get_args

!*******************************************************************************!
! Subroutine:  am_tor_tor_func
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_tor_tor_func(list, arg1, arg2, tortor_table,  &
                           func, dfunc_darg1, dfunc_darg2)

  implicit none

! Formal arguments:

  integer, intent(in)                     :: list(6, *) !5 atoms plus param ptr

  double precision, intent(in)            :: arg1(*)
  double precision, intent(in)            :: arg2(*)
  type(angle_angle_functable), intent(in) :: tortor_table(*)
  double precision, intent(out)           :: func(*)
  double precision, intent(out)           :: dfunc_darg1(*)
  double precision, intent(out)           :: dfunc_darg2(*)

! Local variables:

  integer                                 :: am_val_real_array_index
  integer                                 :: n, it, ind1, ind2
  double precision                        :: ang1, ang2
  double precision                        :: ang1_lo, ang1_hi
  double precision                        :: ang2_lo, ang2_hi
  double precision                        :: f(4), e
  double precision                        :: df_da1(4), df_da2(4)
  double precision                        :: d2f_da1_da2(4)
  double precision                        :: de_dang1, de_dang2

  do n = 1, num_list

    it = list(6, n)
    ang1 = arg1(n)
    ang2 = arg2(n)

    ind1 = am_val_real_array_index(ang1, tortor_table(it)%angle1, &
                                  tortor_table(it)%dim1) 

    ind2 = am_val_real_array_index(ang2, tortor_table(it)%angle1, &
                                  tortor_table(it)%dim1) 

    ang1_lo = tortor_table(it)%angle1(ind1)
    ang1_hi = tortor_table(it)%angle1(ind1 + 1)
    ang2_lo = tortor_table(it)%angle2(ind2)
    ang2_hi = tortor_table(it)%angle2(ind2 + 1)

    ! counter-clockwise order around surrounding table vertices

    f(1) = tortor_table(it)%func(ind1, ind2)
    f(2) = tortor_table(it)%func(ind1 + 1, ind2)
    f(3) = tortor_table(it)%func(ind1 + 1, ind2 + 1)
    f(4) = tortor_table(it)%func(ind1, ind2 + 1)

    df_da1(1) = tortor_table(it)%dfunc_dangle1(ind1, ind2)
    df_da1(2) = tortor_table(it)%dfunc_dangle1(ind1 + 1, ind2)
    df_da1(3) = tortor_table(it)%dfunc_dangle1(ind1 + 1, ind2 + 1)
    df_da1(4) = tortor_table(it)%dfunc_dangle1(ind1, ind2 + 1)

    df_da2(1) = tortor_table(it)%dfunc_dangle2(ind1, ind2)
    df_da2(2) = tortor_table(it)%dfunc_dangle2(ind1 + 1, ind2)
    df_da2(3) = tortor_table(it)%dfunc_dangle2(ind1 + 1, ind2 + 1)
    df_da2(4) = tortor_table(it)%dfunc_dangle2(ind1, ind2 + 1)

    d2f_da1_da2(1) = tortor_table(it)%d2func_dangle1_dangle2(ind1, ind2)
    d2f_da1_da2(2) = tortor_table(it)%d2func_dangle1_dangle2(ind1 + 1, ind2)
    d2f_da1_da2(3) = tortor_table(it)%d2func_dangle1_dangle2(ind1 + 1, ind2 + 1)
    d2f_da1_da2(4) = tortor_table(it)%d2func_dangle1_dangle2(ind1, ind2 + 1)

    call am_val_bcuint1(f, df_da1, df_da2, d2f_da1_da2, ang1_lo, ang1_hi, &
                        ang2_lo, ang2_hi, ang1, ang2, e, de_dang1, de_dang2)
    func(n) = e
    dfunc_darg1(n) = de_dang1
    dfunc_darg2(n) = de_dang2

  end do

  return

end subroutine am_tor_tor_func

!*******************************************************************************!
! Subroutine:  am_tor_tor_get_ene_frc
!
! Description: <TBS>
!
!*******************************************************************************

subroutine am_tor_tor_get_ene_frc(list, crd, fn, dfn_darg1, dfn_darg2, &
                                  darg1_dcrdijkl, darg2_dcrdjklm, frc)

  implicit none

! Formal arguments:

  integer, intent(in)                   :: list(6, *) !5 atoms plus param ptr
  double precision, intent(in)          :: crd(3, *)
  double precision, intent(in)          :: fn(*)
  double precision, intent(in)          :: dfn_darg1(*)
  double precision, intent(in)          :: dfn_darg2(*)
  double precision, intent(in)          :: darg1_dcrdijkl(12, *)
  double precision, intent(in)          :: darg2_dcrdjklm(12, *)
  double precision, intent(in out)      :: frc(3, *)

! Local variables:

  integer                               ::  i, j, k, l, m, n, p, q
  double precision                      :: f(12), g(12), term

  do n = 1, num_list

    i = list(1, n)
    j = list(2, n)
    k = list(3, n)
    l = list(4, n)
    m = list(5, n)

    energy = energy + fn(n)

! Apply chain rule to get deriv of energy with respect to crds of i, j, k, l, m.
! dfn_darg1 holds the deriv of fn with respect to arg1.
! dfn_darg2 holds the deriv of fn with respect to arg2
! while darg1_dcrdijkl holds the derivs of arg1 with respect to crds of
! i, j, k, l and darg2_dcrdjklm holds the derivs of arg2 with respect to crds of
! j, k, l, m.
! Recall force is negative of grad.

    term = dfn_darg1(n)

    do p = 1, 12
      f(p) = term * darg1_dcrdijkl(p, n)
    end do

    term = dfn_darg2(n)

    do p = 1, 12
      g(p) = term * darg2_dcrdjklm(p, n)
    end do

    do p = 1, 3
      frc(p, i) = frc(p, i) - f(p)
      frc(p, j) = frc(p, j) - f(p + 3) - g(p)
      frc(p, k) = frc(p, k) - f(p + 6) - g(p + 3)
      frc(p, l) = frc(p, l) - f(p + 9) - g(p + 6)
      frc(p, m) = frc(p, m) - g(p + 9)
    end do

! Now get virial.

    do q = 1, 3
      do p = 1, 3
        virial(p, q) = virial(p, q) + f(p) * crd(q, i) + &
                                      f(p + 3) * crd(q, j) + &
                                      f(p + 6) * crd(q, k) + &
                                      f(p + 9) * crd(q, l) + &
                                      g(p) * crd(q, j) + &
                                      g(p + 3) * crd(q, k) + &
                                      g(p + 6) * crd(q, l) + &
                                      g(p + 9) * crd(q, m)
      end do
    end do

  end do

  return

end subroutine am_tor_tor_get_ene_frc

#endif /* AMOEBA */
end module amoeba_torsion_torsion_mod
